/**
 * Zod Schemas for Favorites API
 */

import { z } from "zod";
import { AddressDBSchema } from "@/lib/schemas/home";

export const FavoriteServiceSchema = z.object({
  id: z.number(),
  serviceName: z.string(),
  price: z.string(),
  discount: z.number(),
});

export const FavoriteVenueSchema = z.object({
  id: z.number(),
  salonName: z.string(),
  coverImage: z.string().nullable(),
  isExternalEmployee: z.boolean(),
  isOnline: z.boolean(),
  // API can return rating as string or number, coerce to number
  salonAverageRating: z
    .any()
    .transform((val) => {
      if (val === null || val === undefined || val === "" || val === "null") {
        return null;
      }
      if (typeof val === "number") {
        return val;
      }
      if (typeof val === "string") {
        const parsed = parseFloat(val);
        return isNaN(parsed) ? null : parsed;
      }
      return null;
    })
    .pipe(z.number().nullable()),
  ratingCount: z.number(),
  distance: z.number(),
  addressDB: AddressDBSchema,
  services: z.array(FavoriteServiceSchema),
  user: z.object({
    id: z.number(),
  }),
  like: z.boolean(),
  averageDiscount: z.number(),
});

export const GetFavoritesRequestSchema = z.object({
  targetLat: z.number(),
  targetLng: z.number(),
  rangeInKm: z.number().min(0),
});

export const GetFavoritesResponseSchema = z.object({
  slons: z.array(FavoriteVenueSchema), // Note: API uses "slons" (typo)
  currencyUnit: z.string(),
  distanceUnit: z.string(),
  highestPriceAcrossAllSalons: z.string(),
  lowestPriceAcrossAllSalons: z.string(),
});

// Like/Dislike Salon Request
export const LikeDislikeSalonRequestSchema = z.object({
  salonId: z.union([z.number(), z.string().transform(Number)]),
});

// Like/Dislike Salon Response
export const LikeDislikeSalonResponseSchema = z.object({
  status: z.string().optional(),
  message: z.string().optional(),
  data: z.object({
    like: z.boolean(),
  }).optional(),
  error: z.string().nullable().optional(),
}).passthrough(); // Allow additional fields

export type FavoriteService = z.infer<typeof FavoriteServiceSchema>;
export type FavoriteVenue = z.infer<typeof FavoriteVenueSchema>;
export type GetFavoritesRequest = z.infer<typeof GetFavoritesRequestSchema>;
export type GetFavoritesResponse = z.infer<typeof GetFavoritesResponseSchema>;
export type LikeDislikeSalonRequest = z.infer<typeof LikeDislikeSalonRequestSchema>;
export type LikeDislikeSalonResponse = z.infer<typeof LikeDislikeSalonResponseSchema>;
