/**
 * Client-side API Fetcher
 * Use in Client Components, React Query hooks
 */

"use client";

import { API_BASE_URL } from "@/config/env";
import { getClientToken } from "@/lib/api/auth-client";

export interface ApiResponse<T = any> {
  status: string;
  message: string;
  data: T;
  error: string;
}

export class ApiError extends Error {
  constructor(
    public status: string,
    public message: string,
    public statusCode?: number,
    public error?: string
  ) {
    super(message);
    this.name = "ApiError";
  }
}

export interface FetcherOptions extends RequestInit {
  requireAuth?: boolean;
  headers?: HeadersInit;
}

/**
 * Normalize API response data to camelCase
 */
function normalizeToCamelCase(obj: any): any {
  if (obj === null || obj === undefined) {
    return obj;
  }

  if (Array.isArray(obj)) {
    return obj.map(normalizeToCamelCase);
  }

  if (typeof obj !== "object") {
    return obj;
  }

  const normalized: any = {};
  for (const [key, value] of Object.entries(obj)) {
    const camelKey = key.replace(/_([a-z])/g, (_, letter) => letter.toUpperCase());
    normalized[camelKey] = normalizeToCamelCase(value);
  }

  return normalized;
}

/**
 * Client-side fetcher
 * Reads tokens from localStorage
 */
export async function clientFetcher<T = any>(
  endpoint: string,
  options: FetcherOptions = {}
): Promise<T> {
  const {
    requireAuth = true,
    headers: customHeaders = {},
    ...fetchOptions
  } = options;

  // Ensure endpoint starts with / and API_BASE_URL doesn't end with /
  const cleanEndpoint = endpoint.startsWith("/") ? endpoint : `/${endpoint}`;
  const cleanBaseUrl = API_BASE_URL.endsWith("/") 
    ? API_BASE_URL.slice(0, -1) 
    : API_BASE_URL;
  const url = `${cleanBaseUrl}${cleanEndpoint}`;

  // Log the full URL for debugging (remove in production)
  if (process.env.NODE_ENV === "development") {
    console.log("[clientFetcher] Full URL:", url);
    console.log("[clientFetcher] Method:", fetchOptions.method || "GET");
  }

  // Build headers
  const headersObj: Record<string, string> = {
    "Content-Type": "application/json",
    Accept: "application/json",
  };

  // Add custom headers if provided as a plain object
  if (customHeaders && typeof customHeaders === "object" && !Array.isArray(customHeaders) && !(customHeaders instanceof Headers)) {
    Object.assign(headersObj, customHeaders);
  }

  // Add authentication token from localStorage
  if (requireAuth) {
    const token = getClientToken();
    
    if (token) {
      headersObj.accessToken = token;
    }
  }

  // Add device token
  try {
    const { getDeviceToken } = await import("@/lib/deviceToken");
    const deviceToken = getDeviceToken();
    if (deviceToken) {
      headersObj["X-Device-Token"] = deviceToken;
    }
  } catch (error) {
    // Device token is optional
  }

  // Convert to HeadersInit for fetch
  const headers: HeadersInit = headersObj;

  try {
    const response = await fetch(url, {
      ...fetchOptions,
      headers,
    });

    // Handle non-JSON responses
    const contentType = response.headers.get("content-type");
    if (!contentType?.includes("application/json")) {
      throw new ApiError(
        "0",
        `Unexpected response type: ${contentType}`,
        response.status
      );
    }

    const data: ApiResponse<T> = await response.json();

    // Log response for debugging (remove in production)
    if (process.env.NODE_ENV === "development") {
      console.log("[clientFetcher] Response status:", response.status);
      console.log("[clientFetcher] Response data:", data);
    }

    // Check if status indicates an error
    if (data.status !== "1" && data.status !== "2") {
      // Log error details
      console.error("[clientFetcher] API Error:", {
        status: data.status,
        message: data.message,
        error: data.error,
        httpStatus: response.status,
      });
      
      throw new ApiError(
        data.status,
        data.message || data.error || "An error occurred",
        response.status,
        data.error
      );
    }

    // Normalize and return data
    const normalizedData = normalizeToCamelCase(data.data) as T;
    return normalizedData;
  } catch (error) {
    // Re-throw ApiError as-is
    if (error instanceof ApiError) {
      throw error;
    }

    // Handle network errors
    if (error instanceof TypeError && error.message === "Failed to fetch") {
      throw new ApiError(
        "0",
        "Network error. Please check your connection.",
        0
      );
    }

    // Handle other errors
    throw new ApiError(
      "0",
      error instanceof Error ? error.message : "An unexpected error occurred",
      500
    );
  }
}


