"use client";

import { useState, useEffect, useRef } from "react";
import { Clock, ChevronDown } from "lucide-react";

interface TimePickerProps {
  isOpen: boolean;
  onClose: () => void;
  onSelect: (time: {
    period?: "any" | "morning" | "afternoon" | "evening";
    from?: string;
    to?: string;
  }) => void;
  selectedTime: {
    period?: "any" | "morning" | "afternoon" | "evening";
    from?: string;
    to?: string;
  } | null;
}

const timeOptions = [
  "12:00 am",
  "12:30 am",
  "1:00 am",
  "1:30 am",
  "2:00 am",
  "2:30 am",
  "3:00 am",
  "3:30 am",
  "4:00 am",
  "4:30 am",
  "5:00 am",
  "5:30 am",
  "6:00 am",
  "6:30 am",
  "7:00 am",
  "7:30 am",
  "8:00 am",
  "8:30 am",
  "9:00 am",
  "9:30 am",
  "10:00 am",
  "10:30 am",
  "11:00 am",
  "11:30 am",
  "12:00 pm",
  "12:30 pm",
  "1:00 pm",
  "1:30 pm",
  "2:00 pm",
  "2:30 pm",
  "3:00 pm",
  "3:30 pm",
  "4:00 pm",
  "4:30 pm",
  "5:00 pm",
  "5:30 pm",
  "6:00 pm",
  "6:30 pm",
  "7:00 pm",
  "7:30 pm",
  "8:00 pm",
  "8:30 pm",
  "9:00 pm",
  "9:30 pm",
  "10:00 pm",
  "10:30 pm",
  "11:00 pm",
  "11:30 pm",
];

const defaultTimeRanges = {
  morning: { from: "6:00 am", to: "12:00 pm" },
  afternoon: { from: "12:00 pm", to: "6:00 pm" },
  evening: { from: "6:00 pm", to: "11:00 pm" },
};

export function TimePicker({
  isOpen,
  onClose,
  onSelect,
  selectedTime,
}: TimePickerProps) {
  const [period, setPeriod] = useState<
    "any" | "morning" | "afternoon" | "evening"
  >(selectedTime?.period || "any");
  const [fromTime, setFromTime] = useState(selectedTime?.from || "6:00 am");
  const [toTime, setToTime] = useState(selectedTime?.to || "12:00 pm");
  const [fromDropdownOpen, setFromDropdownOpen] = useState(false);
  const [toDropdownOpen, setToDropdownOpen] = useState(false);
  const dropdownRef = useRef<HTMLDivElement>(null);
  const fromDropdownRef = useRef<HTMLDivElement>(null);
  const toDropdownRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (
        dropdownRef.current &&
        !dropdownRef.current.contains(event.target as Node) &&
        fromDropdownRef.current &&
        !fromDropdownRef.current.contains(event.target as Node) &&
        toDropdownRef.current &&
        !toDropdownRef.current.contains(event.target as Node)
      ) {
        onClose();
      }
    };

    if (isOpen) {
      document.addEventListener("mousedown", handleClickOutside);
    }

    return () => {
      document.removeEventListener("mousedown", handleClickOutside);
    };
  }, [isOpen, onClose]);

  useEffect(() => {
    if (period !== "any" && defaultTimeRanges[period]) {
      const range = defaultTimeRanges[period];
      setFromTime(range.from);
      setToTime(range.to);
      // Auto-select when period changes
      onSelect({
        period: period,
        from: range.from,
        to: range.to,
      });
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [period]);

  const handlePeriodSelect = (
    selectedPeriod: "any" | "morning" | "afternoon" | "evening"
  ) => {
    setPeriod(selectedPeriod);
    if (selectedPeriod === "any") {
      onSelect({ period: "any" });
      onClose();
    } else {
      const range = defaultTimeRanges[selectedPeriod];
      setFromTime(range.from);
      setToTime(range.to);
    }
  };

  useEffect(() => {
    // Auto-update when time range changes manually (but not on initial period set)
    if (period !== "any" && fromTime && toTime) {
      const timer = setTimeout(() => {
        onSelect({
          period: period,
          from: fromTime,
          to: toTime,
        });
      }, 300);
      return () => clearTimeout(timer);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [fromTime, toTime]);

  const handleTimeRangeSelect = () => {
    onSelect({
      period: period === "any" ? undefined : period,
      from: fromTime,
      to: toTime,
    });
    onClose();
  };

  const getDisplayText = () => {
    if (!selectedTime || selectedTime.period === "any") return "Any time";
    if (selectedTime.period) {
      return `${selectedTime.from || "6:00 am"} - ${
        selectedTime.to || "12:00 pm"
      }`;
    }
    if (selectedTime.from && selectedTime.to) {
      return `${selectedTime.from} - ${selectedTime.to}`;
    }
    return "Any time";
  };

  if (!isOpen) return null;

  return (
    <div
      ref={dropdownRef}
      className="absolute top-full left-0 mt-2 bg-white rounded-xl shadow-lg border border-gray-200 z-50 p-4"
      style={{ width: "312px" }}
    >
      {/* Time Period Buttons */}
      <div className="flex gap-2 mb-4">
        <button
          onClick={() => handlePeriodSelect("any")}
          className={`px-4 py-2 rounded-full text-sm font-medium transition-colors ${
            period === "any"
              ? "bg-purple-600 text-white"
              : "bg-white border border-gray-300 text-gray-700 hover:bg-gray-50"
          }`}
        >
          Any time
        </button>
        <button
          onClick={() => handlePeriodSelect("morning")}
          className={`px-4 py-2 rounded-full text-sm font-medium transition-colors ${
            period === "morning"
              ? "bg-purple-600 text-white"
              : "bg-white border border-gray-300 text-gray-700 hover:bg-gray-50"
          }`}
        >
          Morning
        </button>
        <button
          onClick={() => handlePeriodSelect("afternoon")}
          className={`px-4 py-2 rounded-full text-sm font-medium transition-colors ${
            period === "afternoon"
              ? "bg-purple-600 text-white"
              : "bg-white border border-gray-300 text-gray-700 hover:bg-gray-50"
          }`}
        >
          Afternoon
        </button>
        <button
          onClick={() => handlePeriodSelect("evening")}
          className={`px-4 py-2 rounded-full text-sm font-medium transition-colors ${
            period === "evening"
              ? "bg-purple-600 text-white"
              : "bg-white border border-gray-300 text-gray-700 hover:bg-gray-50"
          }`}
        >
          Evening
        </button>
      </div>

      {/* Time Range Selectors */}
      {period !== "any" && (
        <div className="flex gap-3 items-center mb-2">
          <div className="relative flex-1" ref={fromDropdownRef}>
            <button
              onClick={() => {
                setFromDropdownOpen(!fromDropdownOpen);
                setToDropdownOpen(false);
              }}
              className="w-full px-4 py-2 border border-gray-300 rounded-lg text-left flex items-center justify-between hover:bg-gray-50"
            >
              <span className="body-text-sm text-gray-500">From</span>
              <div className="flex items-center gap-2">
                <span className="body-text">{fromTime}</span>
                <ChevronDown className="w-4 h-4 text-gray-400" />
              </div>
            </button>
            {fromDropdownOpen && (
              <div className="absolute top-full left-0 right-0 mt-1 bg-white border border-gray-200 rounded-lg shadow-lg max-h-60 overflow-y-auto z-10">
                {timeOptions.map((time) => (
                  <button
                    key={time}
                    onClick={() => {
                      setFromTime(time);
                      setFromDropdownOpen(false);
                    }}
                    className={`w-full px-4 py-2 text-left hover:bg-gray-50 ${
                      fromTime === time ? "bg-purple-50" : ""
                    }`}
                  >
                    <span className="body-text-sm">{time}</span>
                  </button>
                ))}
              </div>
            )}
          </div>

          <div className="relative flex-1" ref={toDropdownRef}>
            <button
              onClick={() => {
                setToDropdownOpen(!toDropdownOpen);
                setFromDropdownOpen(false);
              }}
              className="w-full px-4 py-2 border border-gray-300 rounded-lg text-left flex items-center justify-between hover:bg-gray-50"
            >
              <span className="body-text-sm text-gray-500">To</span>
              <div className="flex items-center gap-2">
                <span className="body-text">{toTime}</span>
                <ChevronDown className="w-4 h-4 text-gray-400" />
              </div>
            </button>
            {toDropdownOpen && (
              <div className="absolute top-full left-0 right-0 mt-1 bg-white border border-gray-200 rounded-lg shadow-lg max-h-60 overflow-y-auto z-10">
                {timeOptions.map((time) => (
                  <button
                    key={time}
                    onClick={() => {
                      setToTime(time);
                      setToDropdownOpen(false);
                    }}
                    className={`w-full px-4 py-2 text-left hover:bg-gray-50 ${
                      toTime === time ? "bg-purple-50" : ""
                    }`}
                  >
                    <span className="body-text-sm">{time}</span>
                  </button>
                ))}
              </div>
            )}
          </div>
        </div>
      )}
    </div>
  );
}
