"use client";

import { useState, useEffect, useRef } from "react";
import { Calendar, ChevronLeft, ChevronRight, X } from "lucide-react";

interface DatePickerProps {
  isOpen: boolean;
  onClose: () => void;
  onSelect: (date: { type: "any" | "today" | "tomorrow" | "date"; date?: Date }) => void;
  selectedDate: { type: "any" | "today" | "tomorrow" | "date"; date?: Date } | null;
}

export function DatePicker({
  isOpen,
  onClose,
  onSelect,
  selectedDate,
}: DatePickerProps) {
  const [currentMonth, setCurrentMonth] = useState(new Date());
  const [selectedDay, setSelectedDay] = useState<Date | null>(
    selectedDate?.type === "date" ? selectedDate.date || null : null
  );
  const dropdownRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (
        dropdownRef.current &&
        !dropdownRef.current.contains(event.target as Node)
      ) {
        onClose();
      }
    };

    if (isOpen) {
      document.addEventListener("mousedown", handleClickOutside);
    }

    return () => {
      document.removeEventListener("mousedown", handleClickOutside);
    };
  }, [isOpen, onClose]);

  const today = new Date();
  const tomorrow = new Date(today);
  tomorrow.setDate(tomorrow.getDate() + 1);

  const getDaysInMonth = (date: Date) => {
    const year = date.getFullYear();
    const month = date.getMonth();
    const firstDay = new Date(year, month, 1);
    const lastDay = new Date(year, month + 1, 0);
    const daysInMonth = lastDay.getDate();
    const startingDayOfWeek = firstDay.getDay();
    
    // Adjust to start from Monday (0 = Monday)
    const adjustedStartingDay = startingDayOfWeek === 0 ? 6 : startingDayOfWeek - 1;

    return { daysInMonth, startingDayOfWeek: adjustedStartingDay };
  };

  const { daysInMonth, startingDayOfWeek } = getDaysInMonth(currentMonth);

  const handleQuickSelect = (type: "any" | "today" | "tomorrow") => {
    if (type === "any") {
      onSelect({ type: "any" });
    } else if (type === "today") {
      onSelect({ type: "today", date: today });
    } else if (type === "tomorrow") {
      onSelect({ type: "tomorrow", date: tomorrow });
    }
    onClose();
  };

  const handleDateSelect = (day: number) => {
    const selected = new Date(
      currentMonth.getFullYear(),
      currentMonth.getMonth(),
      day
    );
    setSelectedDay(selected);
    onSelect({ type: "date", date: selected });
    onClose();
  };

  const isToday = (day: number) => {
    const date = new Date(
      currentMonth.getFullYear(),
      currentMonth.getMonth(),
      day
    );
    return date.toDateString() === today.toDateString();
  };

  const isTomorrow = (day: number) => {
    const date = new Date(
      currentMonth.getFullYear(),
      currentMonth.getMonth(),
      day
    );
    return date.toDateString() === tomorrow.toDateString();
  };

  const isSelected = (day: number) => {
    if (!selectedDay) return false;
    const date = new Date(
      currentMonth.getFullYear(),
      currentMonth.getMonth(),
      day
    );
    return date.toDateString() === selectedDay.toDateString();
  };

  const monthNames = [
    "January", "February", "March", "April", "May", "June",
    "July", "August", "September", "October", "November", "December"
  ];

  const dayNames = ["Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"];

  const navigateMonth = (direction: "prev" | "next") => {
    setCurrentMonth((prev) => {
      const newDate = new Date(prev);
      if (direction === "prev") {
        newDate.setMonth(prev.getMonth() - 1);
      } else {
        newDate.setMonth(prev.getMonth() + 1);
      }
      return newDate;
    });
  };

  if (!isOpen) return null;

  return (
    <div
      ref={dropdownRef}
      className="bg-white rounded-2xl shadow-xl border border-gray-200 z-50 p-6"
    >
      <div className="flex flex-col lg:flex-row gap-6">
        {/* Left Side - Quick Select */}
        <div className="flex flex-row lg:flex-col gap-3">
          <button
            onClick={() => handleQuickSelect("today")}
            className={`px-4 py-3 rounded-xl text-sm font-medium transition-all text-left lg:min-w-[160px] ${
              selectedDate?.type === "today"
                ? "bg-purple-600 text-white border-2 border-purple-600"
                : "bg-white border-2 border-gray-200 text-gray-700 hover:border-gray-300"
            }`}
          >
            <div className="font-semibold">Today</div>
            <div className={`text-xs ${selectedDate?.type === "today" ? "text-purple-100" : "text-gray-500"}`}>
              {today.toLocaleDateString("en-US", { weekday: "short", day: "numeric", month: "short" })}
            </div>
          </button>
          <button
            onClick={() => handleQuickSelect("tomorrow")}
            className={`px-4 py-3 rounded-xl text-sm font-medium transition-all text-left lg:min-w-[160px] ${
              selectedDate?.type === "tomorrow"
                ? "bg-purple-600 text-white border-2 border-purple-600"
                : "bg-white border-2 border-gray-200 text-gray-700 hover:border-gray-300"
            }`}
          >
            <div className="font-semibold">Tomorrow</div>
            <div className={`text-xs ${selectedDate?.type === "tomorrow" ? "text-purple-100" : "text-gray-500"}`}>
              {tomorrow.toLocaleDateString("en-US", { weekday: "short", day: "numeric", month: "short" })}
            </div>
          </button>
        </div>

        {/* Right Side - Calendar */}
        <div className="flex-1">
          {/* Month Navigation */}
          <div className="flex items-center justify-between mb-4">
            <button
              onClick={() => navigateMonth("prev")}
              className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
            >
              <ChevronLeft className="w-5 h-5 text-gray-600" />
            </button>
            <h3 className="text-lg font-semibold text-gray-900">
              {currentMonth.toLocaleDateString("en-US", { month: "long", year: "numeric" })}
            </h3>
            <button
              onClick={() => navigateMonth("next")}
              className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
            >
              <ChevronRight className="w-5 h-5 text-gray-600" />
            </button>
          </div>

          {/* Calendar Grid */}
          <div className="grid grid-cols-7 gap-1 mb-2">
            {dayNames.map((day) => (
              <div
                key={day}
                className="text-center text-xs font-medium text-gray-500 py-2"
              >
                {day}
              </div>
            ))}
          </div>

          <div className="grid grid-cols-7 gap-1">
            {Array.from({ length: startingDayOfWeek }).map((_, index) => (
              <div key={`empty-${index}`} className="aspect-square" />
            ))}
            {Array.from({ length: daysInMonth }).map((_, index) => {
              const day = index + 1;
              const isSelectedDay = isSelected(day);
              const isTodayDay = isToday(day);

              return (
                <button
                  key={day}
                  onClick={() => handleDateSelect(day)}
                  className={`aspect-square rounded-full text-sm font-medium transition-all ${
                    isSelectedDay
                      ? "bg-purple-600 text-white"
                      : isTodayDay
                      ? "bg-purple-50 text-purple-700 border border-purple-200"
                      : "text-gray-700 hover:bg-gray-50"
                  }`}
                >
                  {day}
                </button>
              );
            })}
          </div>
        </div>
      </div>
    </div>
  );
}

