/**
 * Search Venues API
 * Searches venues by query, location, filters
 */

import { clientFetcher } from "@/lib/api/fetcher-client";
import { VenueSchema, type Venue } from "@/lib/schemas/venue";
import { z } from "zod";

const SearchVenuesResponseSchema = z.object({
  venues: z.array(VenueSchema),
  total: z.number().int(),
  page: z.number().int().optional(),
  limit: z.number().int().optional(),
});

export interface SearchVenuesParams {
  query?: string;
  location?: string;
  category?: string;
  minRating?: number;
  page?: number;
  limit?: number;
  sortBy?: "relevance" | "rating" | "distance" | "newest";
}

/**
 * Server-side: Search venues
 * Dynamically imports server fetcher to avoid bundling in client code
 */
export async function searchVenuesServer(
  params: SearchVenuesParams
): Promise<{ venues: Venue[]; total: number }> {
  // Dynamically import server fetcher to avoid bundling in client
  const { serverFetcher } = await import("@/lib/api/fetcher-server");

  const queryParams = new URLSearchParams();
  if (params.query) queryParams.set("query", params.query);
  if (params.location) queryParams.set("location", params.location);
  if (params.category) queryParams.set("category", params.category);
  if (params.minRating) queryParams.set("minRating", params.minRating.toString());
  if (params.page) queryParams.set("page", params.page.toString());
  if (params.limit) queryParams.set("limit", params.limit.toString());
  if (params.sortBy) queryParams.set("sortBy", params.sortBy);

  const endpoint = `/venues/search${
    queryParams.toString() ? `?${queryParams.toString()}` : ""
  }`;

  const data = await serverFetcher<unknown>(endpoint, {
    requireAuth: false,
    method: "GET",
  });

  const parsed = SearchVenuesResponseSchema.parse(data);
  return {
    venues: parsed.venues,
    total: parsed.total,
  };
}

/**
 * Client-side: Search venues
 */
export async function searchVenuesClient(
  params: SearchVenuesParams
): Promise<{ venues: Venue[]; total: number }> {
  const queryParams = new URLSearchParams();
  if (params.query) queryParams.set("query", params.query);
  if (params.location) queryParams.set("location", params.location);
  if (params.category) queryParams.set("category", params.category);
  if (params.minRating) queryParams.set("minRating", params.minRating.toString());
  if (params.page) queryParams.set("page", params.page.toString());
  if (params.limit) queryParams.set("limit", params.limit.toString());
  if (params.sortBy) queryParams.set("sortBy", params.sortBy);

  const endpoint = `/venues/search${
    queryParams.toString() ? `?${queryParams.toString()}` : ""
  }`;

  const data = await clientFetcher<unknown>(endpoint, {
    requireAuth: false,
    method: "GET",
  });

  const parsed = SearchVenuesResponseSchema.parse(data);
  return {
    venues: parsed.venues,
    total: parsed.total,
  };
}
