"use client";

import Image from "next/image";
import { useState, useRef, useEffect } from "react";
import { PlaceholderImage } from "@/components/data-display/PlaceholderImage";
import { SalonImage, SalonDetail } from "@/lib/schemas/salonDetail";
import { getImageUrl } from "../utils/imageUtils";
import { ChevronLeft, ChevronRight } from "lucide-react";

interface SalonImageGalleryProps {
  salon: SalonDetail;
}

export function SalonImageGallery({ salon }: SalonImageGalleryProps) {
  const [selectedIndex, setSelectedIndex] = useState(0);
  const [imageErrors, setImageErrors] = useState<Set<string>>(new Set());
  const scrollContainerRef = useRef<HTMLDivElement>(null);
  const carouselRef = useRef<HTMLDivElement>(null);
  const [touchStart, setTouchStart] = useState(0);
  const [touchEnd, setTouchEnd] = useState(0);
  const [isOverflowed, setIsOverflowed] = useState(false);
  const [canScrollLeft, setCanScrollLeft] = useState(false);
  const [canScrollRight, setCanScrollRight] = useState(false);

  // Get cover image as first image
  const coverImage = salon.coverImage;
  
  // Get salon images
  const salonImages = salon.salonImages || [];
  const salonImageUrls = salonImages
    .map((img) => img.imageUrl)
    .filter((url): url is string => Boolean(url));

  // Combine: cover image first, then salon images
  const allImages: (string | null)[] = [];
  if (coverImage) {
    allImages.push(coverImage);
  }
  salonImageUrls.forEach((url) => {
    if (url && !allImages.includes(url)) {
      allImages.push(url);
    }
  });

  if (allImages.length === 0) {
    return (
      <div className="w-full h-64 md:h-96 rounded-lg overflow-hidden">
        <PlaceholderImage className="w-full h-full" />
      </div>
    );
  }

  const mainImage = allImages[selectedIndex];
  const hasMultipleImages = allImages.length > 1;
  const hasMoreThanFour = salonImageUrls.length > 4;

  const scrollLeft = () => {
    if (scrollContainerRef.current) {
      scrollContainerRef.current.scrollBy({ left: -200, behavior: "smooth" });
      // Check overflow after scroll animation
      setTimeout(() => checkOverflow(), 300);
    }
  };

  const scrollRight = () => {
    if (scrollContainerRef.current) {
      scrollContainerRef.current.scrollBy({ left: 200, behavior: "smooth" });
      // Check overflow after scroll animation
      setTimeout(() => checkOverflow(), 300);
    }
  };

  // Check if carousel is overflowed and update scroll button states
  const checkOverflow = () => {
    if (scrollContainerRef.current) {
      const container = scrollContainerRef.current;
      const isOverflow = container.scrollWidth > container.clientWidth;
      setIsOverflowed(isOverflow);
      
      if (isOverflow) {
        setCanScrollLeft(container.scrollLeft > 0);
        setCanScrollRight(
          container.scrollLeft < container.scrollWidth - container.clientWidth - 1
        );
      } else {
        setCanScrollLeft(false);
        setCanScrollRight(false);
      }
    }
  };

  // Get other images (excluding the first/cover image) for desktop carousel
  const otherImages = allImages.slice(1);
  const hasOtherImages = otherImages.length > 0;
  
  // For desktop carousel, include cover image as first item so users can click back to it
  const desktopCarouselImages = coverImage ? [coverImage, ...otherImages] : otherImages;

  // Check overflow on mount and when images change
  useEffect(() => {
    // Use setTimeout to ensure DOM is rendered
    const timer = setTimeout(() => {
      checkOverflow();
    }, 100);
    
    // Also check on window resize
    const handleResize = () => {
      checkOverflow();
    };
    window.addEventListener("resize", handleResize);
    
    // Check scroll position to update arrow visibility
    const handleScroll = () => {
      checkOverflow();
    };
    
    const container = scrollContainerRef.current;
    if (container) {
      container.addEventListener("scroll", handleScroll);
    }

    return () => {
      clearTimeout(timer);
      window.removeEventListener("resize", handleResize);
      if (container) {
        container.removeEventListener("scroll", handleScroll);
      }
    };
  }, [desktopCarouselImages.length, allImages.length]);

  // Get thumbnail images (salon images only, excluding cover)
  const thumbnailImages = salonImageUrls.slice(0, hasMoreThanFour ? undefined : 4);

  return (
    <div className="space-y-2">
      {/* Mobile & Tablet Layout: Swiper/Carousel */}
      <div className="lg:hidden">
        <div
          ref={carouselRef}
          className="relative w-full max-w-full h-[300px] md:h-[400px] overflow-hidden bg-gray-100 rounded-lg"
          onTouchStart={(e) => setTouchStart(e.targetTouches[0].clientX)}
          onTouchMove={(e) => setTouchEnd(e.targetTouches[0].clientX)}
          onTouchEnd={() => {
            if (!touchStart || !touchEnd) return;
            const distance = touchStart - touchEnd;
            const isLeftSwipe = distance > 50;
            const isRightSwipe = distance < -50;
            if (isLeftSwipe && selectedIndex < allImages.length - 1) {
              setSelectedIndex(selectedIndex + 1);
            }
            if (isRightSwipe && selectedIndex > 0) {
              setSelectedIndex(selectedIndex - 1);
            }
          }}
        >
          {/* Images Container */}
          <div
            className="flex transition-transform duration-300 ease-out h-full"
            style={{
              transform: `translateX(-${selectedIndex * 100}%)`,
            }}
          >
            {allImages.map((imageUrl, index) => (
              <div
                key={imageUrl || `img-${index}`}
                className="relative w-full h-full flex-shrink-0 rounded-lg overflow-hidden"
              >
                {imageUrl && !imageErrors.has(imageUrl) ? (
                  <Image
                    src={getImageUrl(imageUrl)}
                    alt={`${salon.salonName || "Salon"} image ${index + 1}`}
                    fill
                    className="object-cover"
                    onError={() => {
                      setImageErrors((prev) => new Set(prev).add(imageUrl));
                    }}
                  />
                ) : (
                  <PlaceholderImage className="w-full h-full" />
                )}
              </div>
            ))}
          </div>

          {/* Image Counter */}
          {allImages.length > 1 && (
            <div className="absolute bottom-4 right-4 bg-black/60 text-white px-3 py-1.5 rounded-full text-sm font-medium">
              {selectedIndex + 1}/{allImages.length}
            </div>
          )}

          {/* Navigation Arrows */}
          {allImages.length > 1 && (
            <>
              {selectedIndex > 0 && (
                <button
                  onClick={() => setSelectedIndex(selectedIndex - 1)}
                  className="absolute left-4 top-1/2 -translate-y-1/2 w-10 h-10 rounded-full bg-white/90 hover:bg-white shadow-lg flex items-center justify-center transition-colors z-10"
                  aria-label="Previous image"
                >
                  <ChevronLeft className="w-5 h-5 text-gray-900" />
                </button>
              )}
              {selectedIndex < allImages.length - 1 && (
                <button
                  onClick={() => setSelectedIndex(selectedIndex + 1)}
                  className="absolute right-4 top-1/2 -translate-y-1/2 w-10 h-10 rounded-full bg-white/90 hover:bg-white shadow-lg flex items-center justify-center transition-colors z-10"
                  aria-label="Next image"
                >
                  <ChevronRight className="w-5 h-5 text-gray-900" />
                </button>
              )}
            </>
          )}

          {/* Dots Indicator */}
          {allImages.length > 1 && (
            <div className="absolute bottom-4 left-1/2 -translate-x-1/2 flex gap-2 z-10">
              {allImages.map((_, index) => (
                <button
                  key={index}
                  onClick={() => setSelectedIndex(index)}
                  className={`w-2 h-2 rounded-full transition-all ${
                    index === selectedIndex
                      ? "bg-white w-6"
                      : "bg-white/50 hover:bg-white/75"
                  }`}
                  aria-label={`Go to image ${index + 1}`}
                />
              ))}
            </div>
          )}
        </div>
      </div>

      {/* Desktop Layout: First image on top, other images in carousel below */}
      <div className="hidden lg:block">
        {/* Large Image - Shows selected image (cover by default, or clicked carousel image) */}
        <div className="relative w-full h-64 md:h-96 rounded-lg overflow-hidden bg-gray-100 mb-4">
          {mainImage && !imageErrors.has(mainImage) ? (
            <Image
              src={getImageUrl(mainImage)}
              alt={salon.salonName || "Salon image"}
              fill
              className="object-cover"
              onError={() => {
                setImageErrors((prev) => new Set(prev).add(mainImage));
              }}
            />
          ) : (
            <PlaceholderImage className="w-full h-full" />
          )}
        </div>

        {/* Other Images Carousel - Only show if there are other images */}
        {hasOtherImages && (
          <div className="relative">
            {isOverflowed && (
              <>
                {canScrollLeft && (
                  <button
                    onClick={scrollLeft}
                    className="absolute left-0 top-1/2 -translate-y-1/2 z-10 p-1.5 rounded-full bg-white shadow-md hover:bg-gray-50 transition-colors"
                    aria-label="Scroll left"
                  >
                    <ChevronLeft className="w-5 h-5 text-gray-600" />
                  </button>
                )}
                {canScrollRight && (
                  <button
                    onClick={scrollRight}
                    className="absolute right-0 top-1/2 -translate-y-1/2 z-10 p-1.5 rounded-full bg-white shadow-md hover:bg-gray-50 transition-colors"
                    aria-label="Scroll right"
                  >
                    <ChevronRight className="w-5 h-5 text-gray-600" />
                  </button>
                )}
              </>
            )}
            <div
              ref={scrollContainerRef}
              className={`flex gap-2 overflow-x-auto pb-2 scrollbar-hide ${
                desktopCarouselImages.length > 4 ? "" : "grid grid-cols-4"
              }`}
            >
              {desktopCarouselImages.map((imageUrl, index) => {
                const isSelected = selectedIndex === index;

                return (
                  <button
                    key={imageUrl || `carousel-img-${index}`}
                    onClick={() => setSelectedIndex(index)}
                    className={`relative flex-shrink-0 aspect-square rounded-lg overflow-hidden border-2 transition-all ${
                      isSelected
                        ? "border-blue-500"
                        : "border-transparent hover:border-gray-300"
                    } ${desktopCarouselImages.length > 4 ? "w-24" : ""}`}
                  >
                    {imageUrl && !imageErrors.has(imageUrl) ? (
                      <Image
                        src={getImageUrl(imageUrl)}
                        alt={`${salon.salonName || "Salon"} image ${index + 1}`}
                        fill
                        className="object-cover"
                        onError={() => {
                          setImageErrors((prev) => new Set(prev).add(imageUrl));
                        }}
                      />
                    ) : (
                      <PlaceholderImage className="w-full h-full" />
                    )}
                  </button>
                );
              })}
            </div>
          </div>
        )}
      </div>
    </div>
  );
}
