/**
 * Transform HomeSalon from API to Venue format
 */

import { HomeSalon } from "@/lib/schemas/home";
import { Venue } from "@/types/venue";
import { API_BASE_URL } from "@/config/env";

/**
 * Transform HomeSalon from API to Venue format
 */
export function transformSalonToVenue(salon: HomeSalon): Venue {
  const address = salon.addressDB;
  const fullAddress = [
    address.streetAddress,
    address.city,
    address.district,
    address.province,
    address.country,
  ]
    .filter(Boolean)
    .join(", ");

  // Handle image URL - convert relative paths to absolute URLs
  // Use a placeholder URL that can be detected for showing placeholder component
  let imageUrl = "/placeholder-venue";
  if (salon.coverImage) {
    if (salon.coverImage.startsWith("http://") || salon.coverImage.startsWith("https://")) {
      imageUrl = salon.coverImage;
    } else if (salon.coverImage.startsWith("/")) {
      imageUrl = `${API_BASE_URL}${salon.coverImage}`;
    } else {
      // Relative path like "Public/Gallery/..." - prepend base URL with leading slash
      imageUrl = `${API_BASE_URL}/${salon.coverImage}`;
    }
  }

  return {
    id: salon.id.toString(),
    slug: salon.id.toString(), // Use salon ID as slug for API endpoint
    name: salon.salonName,
    image: imageUrl,
    rating: salon.salonAverageRating ?? 0,
    reviewCount: salon.ratingCount,
    category: salon.services[0]?.serviceName || "Service",
    location: address.city,
    address: fullAddress,
    badge: salon.isOnline ? "Online" : undefined,
  };
}

