"use client";

import { useState, useRef, useEffect } from "react";
import Link from "next/link";
import { ChevronLeft, ChevronRight } from "lucide-react";
import { Container } from "@/components/layout/Container";
import { PageSection } from "@/components/layout/PageSection";

const serviceTypes = [
  "Hair Salons",
  "Nails",
  "Spas",
  "Beauty Salons",
  "Barbers",
  "Massage Partners",
  "Waxing Salons",
];

// Country to cities mapping
const countryCitiesMap: Record<
  string,
  Array<{ name: string; services: string[] }>
> = {
  Australia: [
    { name: "Sydney", services: serviceTypes },
    { name: "Melbourne", services: serviceTypes },
    { name: "Perth", services: serviceTypes },
    { name: "Brisbane", services: serviceTypes },
    { name: "Gold Coast", services: serviceTypes },
  ],
  Canada: [
    { name: "Toronto", services: serviceTypes },
    { name: "Vancouver", services: serviceTypes },
    { name: "Montreal", services: serviceTypes },
    { name: "Calgary", services: serviceTypes },
  ],
  France: [
    { name: "Paris", services: serviceTypes },
    { name: "Lyon", services: serviceTypes },
    { name: "Marseille", services: serviceTypes },
    { name: "Toulouse", services: serviceTypes },
  ],
  Germany: [
    { name: "Berlin", services: serviceTypes },
    { name: "Munich", services: serviceTypes },
    { name: "Hamburg", services: serviceTypes },
    { name: "Frankfurt", services: serviceTypes },
  ],
  Italy: [
    { name: "Rome", services: serviceTypes },
    { name: "Milan", services: serviceTypes },
    { name: "Naples", services: serviceTypes },
    { name: "Turin", services: serviceTypes },
  ],
  Netherlands: [
    { name: "Amsterdam", services: serviceTypes },
    { name: "Rotterdam", services: serviceTypes },
    { name: "The Hague", services: serviceTypes },
    { name: "Utrecht", services: serviceTypes },
  ],
  Belgium: [
    { name: "Brussels", services: serviceTypes },
    { name: "Antwerp", services: serviceTypes },
    { name: "Ghent", services: serviceTypes },
  ],
  Brazil: [
    { name: "São Paulo", services: serviceTypes },
    { name: "Rio de Janeiro", services: serviceTypes },
    { name: "Brasília", services: serviceTypes },
  ],
  Denmark: [
    { name: "Copenhagen", services: serviceTypes },
    { name: "Aarhus", services: serviceTypes },
    { name: "Odense", services: serviceTypes },
  ],
  Greece: [
    { name: "Athens", services: serviceTypes },
    { name: "Thessaloniki", services: serviceTypes },
    { name: "Patras", services: serviceTypes },
  ],
  Ireland: [
    { name: "Dublin", services: serviceTypes },
    { name: "Cork", services: serviceTypes },
    { name: "Galway", services: serviceTypes },
  ],
  Mexico: [
    { name: "Mexico City", services: serviceTypes },
    { name: "Guadalajara", services: serviceTypes },
    { name: "Monterrey", services: serviceTypes },
  ],
  Barbados: [
    { name: "Bridgetown", services: serviceTypes },
    { name: "Speightstown", services: serviceTypes },
  ],
};

const countries = Object.keys(countryCitiesMap);

export function CityDirectorySection() {
  // Default to Australia (as shown in screenshot)
  const [activeCountry, setActiveCountry] = useState<string>("Australia");

  // Refs for carousels
  const countryScrollRef = useRef<HTMLDivElement>(null);
  const citiesScrollRef = useRef<HTMLDivElement>(null);

  // State for arrow visibility
  const [countryShowLeft, setCountryShowLeft] = useState(false);
  const [countryShowRight, setCountryShowRight] = useState(true);
  const [citiesShowLeft, setCitiesShowLeft] = useState(false);
  const [citiesShowRight, setCitiesShowRight] = useState(true);

  // Get cities for the active country
  const citiesData = countryCitiesMap[activeCountry] || [];

  // Check scroll position for country carousel
  const checkCountryScroll = () => {
    if (countryScrollRef.current) {
      const { scrollLeft, scrollWidth, clientWidth } = countryScrollRef.current;
      setCountryShowLeft(scrollLeft > 10);
      setCountryShowRight(scrollLeft < scrollWidth - clientWidth - 10);
    }
  };

  // Check scroll position for cities carousel
  const checkCitiesScroll = () => {
    if (citiesScrollRef.current) {
      const { scrollLeft, scrollWidth, clientWidth } = citiesScrollRef.current;
      setCitiesShowLeft(scrollLeft > 10);
      setCitiesShowRight(scrollLeft < scrollWidth - clientWidth - 10);
    }
  };

  // Scroll functions for country carousel
  const scrollCountry = (direction: "left" | "right") => {
    if (countryScrollRef.current) {
      const scrollAmount = 200;
      countryScrollRef.current.scrollBy({
        left: direction === "left" ? -scrollAmount : scrollAmount,
        behavior: "smooth",
      });
    }
  };

  // Scroll functions for cities carousel
  const scrollCities = (direction: "left" | "right") => {
    if (citiesScrollRef.current) {
      const scrollAmount = 400;
      citiesScrollRef.current.scrollBy({
        left: direction === "left" ? -scrollAmount : scrollAmount,
        behavior: "smooth",
      });
    }
  };

  // Set up scroll listeners
  useEffect(() => {
    const countryElement = countryScrollRef.current;
    const citiesElement = citiesScrollRef.current;

    if (countryElement) {
      checkCountryScroll();
      countryElement.addEventListener("scroll", checkCountryScroll);
      // Check on resize
      window.addEventListener("resize", checkCountryScroll);
    }

    if (citiesElement) {
      checkCitiesScroll();
      citiesElement.addEventListener("scroll", checkCitiesScroll);
      // Check on resize
      window.addEventListener("resize", checkCitiesScroll);
    }

    return () => {
      if (countryElement) {
        countryElement.removeEventListener("scroll", checkCountryScroll);
        window.removeEventListener("resize", checkCountryScroll);
      }
      if (citiesElement) {
        citiesElement.removeEventListener("scroll", checkCitiesScroll);
        window.removeEventListener("resize", checkCitiesScroll);
      }
    };
  }, [activeCountry]); // Re-check when country changes

  return (
    <PageSection>
      <Container>
        <div className="py-8 sm:py-12 md:py-16">
          <h2
            className="text-xl sm:text-2xl md:text-3xl lg:text-4xl font-bold mb-4 sm:mb-6 md:mb-8"
            style={{ color: "var(--text-primary)" }}
          >
            Browse by City
          </h2>

          {/* Country Selector Carousel */}
          <div className="relative mb-4 sm:mb-6 md:mb-8">
            {/* Left Arrow - Hidden on mobile */}
            {countryShowLeft && (
              <button
                onClick={() => scrollCountry("left")}
                className="absolute left-0 top-1/2 -translate-y-1/2 z-10 rounded-full p-1.5 sm:p-2 bg-white border border-gray-200 shadow-sm hover:shadow-md transition-all duration-200 hidden md:block"
                style={{
                  backgroundColor: "var(--surface)",
                  border: "1px solid var(--color-gray-300)",
                  boxShadow: "var(--shadow-sm)",
                }}
                aria-label="Scroll countries left"
              >
                <ChevronLeft
                  className="w-3 h-3 sm:w-4 sm:h-4 md:w-5 md:h-5"
                  style={{ color: "var(--text-primary)" }}
                />
              </button>
            )}

            {/* Country Pills Container - Horizontal Scroll */}
            <div
              ref={countryScrollRef}
              className="flex gap-2 overflow-x-auto scrollbar-hide scroll-smooth pb-2 px-1"
              style={{
                scrollbarWidth: "none",
                msOverflowStyle: "none",
                WebkitOverflowScrolling: "touch",
              }}
            >
              {countries.map((country) => (
                <button
                  key={country}
                  onClick={() => setActiveCountry(country)}
                  className="px-3 py-1.5 sm:px-4 sm:py-2 rounded-full text-xs sm:text-sm font-semibold transition-colors whitespace-nowrap flex-shrink-0"
                  style={{
                    backgroundColor:
                      activeCountry === country
                        ? "var(--text-primary)"
                        : "transparent",
                    color:
                      activeCountry === country
                        ? "var(--surface)"
                        : "var(--text-primary)",
                  }}
                >
                  {country}
                </button>
              ))}
            </div>

            {/* Right Arrow - Hidden on mobile */}
            {countryShowRight && (
              <button
                onClick={() => scrollCountry("right")}
                className="absolute right-0 top-1/2 -translate-y-1/2 z-10 rounded-full p-1.5 sm:p-2 bg-white border border-gray-200 shadow-sm hover:shadow-md transition-all duration-200 hidden md:block"
                style={{
                  backgroundColor: "var(--surface)",
                  border: "1px solid var(--color-gray-300)",
                  boxShadow: "var(--shadow-sm)",
                }}
                aria-label="Scroll countries right"
              >
                <ChevronRight
                  className="w-3 h-3 sm:w-4 sm:h-4 md:w-5 md:h-5"
                  style={{ color: "var(--text-primary)" }}
                />
              </button>
            )}
          </div>

          {/* Cities Carousel */}
          <div className="relative">
            {/* Left Arrow */}
            {citiesShowLeft && (
              <button
                onClick={() => scrollCities("left")}
                className="absolute left-0 top-1/2 -translate-y-1/2 z-10 rounded-full p-2 md:p-3 bg-white border border-gray-200 shadow-sm hover:shadow-md transition-all duration-200 hidden md:block"
                style={{
                  backgroundColor: "var(--surface)",
                  border: "1px solid var(--color-gray-300)",
                  boxShadow: "0 1px 2px 0 rgb(0 0 0 / 0.05)",
                }}
                aria-label="Scroll cities left"
              >
                <ChevronLeft
                  className="w-4 h-4 md:w-5 md:h-5"
                  style={{ color: "var(--text-primary)" }}
                />
              </button>
            )}

            {/* Cities Container - Horizontal Scroll */}
            <div
              ref={citiesScrollRef}
              className="flex gap-6 md:gap-8 overflow-x-auto scrollbar-hide scroll-smooth pb-4 items-start"
              style={{ scrollbarWidth: "none", msOverflowStyle: "none" }}
            >
              {citiesData.map((city) => (
                <div
                  key={city.name}
                  className="flex-shrink-0 w-[200px] sm:w-[240px] md:w-[260px] lg:w-[300px]"
                >
                  <Link
                    href={`/city/${city.name
                      .toLowerCase()
                      .replace(/\s+/g, "-")}`}
                    className="text-base sm:text-lg md:text-xl font-semibold block mb-2 sm:mb-3 hover:underline"
                    style={{ color: "var(--text-primary)" }}
                  >
                    {city.name}
                  </Link>
                  <ul className="space-y-1">
                    {city.services.map((service) => (
                      <li key={service}>
                        <Link
                          href={`/city/${city.name
                            .toLowerCase()
                            .replace(/\s+/g, "-")}/${service
                            .toLowerCase()
                            .replace(/\s+/g, "-")}`}
                          className="text-xs sm:text-sm md:text-base block hover:underline"
                          style={{ color: "var(--text-secondary)" }}
                        >
                          {service} in {city.name}
                        </Link>
                      </li>
                    ))}
                  </ul>
                </div>
              ))}
            </div>

            {/* Right Arrow */}
            {citiesShowRight && (
              <button
                onClick={() => scrollCities("right")}
                className="absolute right-0 top-1/2 -translate-y-1/2 z-10 rounded-full p-2 md:p-3 bg-white border border-gray-200 shadow-sm hover:shadow-md transition-all duration-200 hidden md:block"
                style={{
                  backgroundColor: "var(--surface)",
                  border: "1px solid var(--color-gray-300)",
                  boxShadow: "0 1px 2px 0 rgb(0 0 0 / 0.05)",
                }}
                aria-label="Scroll cities right"
              >
                <ChevronRight
                  className="w-4 h-4 md:w-5 md:h-5"
                  style={{ color: "var(--text-primary)" }}
                />
              </button>
            )}
          </div>
        </div>
      </Container>
    </PageSection>
  );
}
