"use client";

import { ReactNode, useRef, useState, useEffect } from "react";
import { ChevronLeft, ChevronRight } from "lucide-react";
import Link from "next/link";

interface HorizontalCarouselProps {
  children: ReactNode;
  title?: string;
  className?: string;
  seeMoreLink?: string;
}

export function HorizontalCarousel({
  children,
  title,
  className = "",
  seeMoreLink,
}: HorizontalCarouselProps) {
  const scrollRef = useRef<HTMLDivElement>(null);
  const [showLeftArrow, setShowLeftArrow] = useState(false);
  const [showRightArrow, setShowRightArrow] = useState(true);

  const checkScrollPosition = () => {
    if (scrollRef.current) {
      const { scrollLeft, scrollWidth, clientWidth } = scrollRef.current;
      setShowLeftArrow(scrollLeft > 10);
      setShowRightArrow(scrollLeft < scrollWidth - clientWidth - 10);
    }
  };

  useEffect(() => {
    const element = scrollRef.current;
    if (element) {
      checkScrollPosition();
      element.addEventListener("scroll", checkScrollPosition);
      // Check on resize
      window.addEventListener("resize", checkScrollPosition);
      return () => {
        element.removeEventListener("scroll", checkScrollPosition);
        window.removeEventListener("resize", checkScrollPosition);
      };
    }
  }, []);

  const scroll = (direction: "left" | "right") => {
    if (scrollRef.current) {
      const scrollAmount = 400;
      scrollRef.current.scrollBy({
        left: direction === "left" ? -scrollAmount : scrollAmount,
        behavior: "smooth",
      });
    }
  };

  return (
    <div className={className}>
      {title && (
        <div className="flex items-center justify-between gap-3 mb-6 sm:mb-8">
          <h2
            className="text-xl sm:text-2xl md:text-3xl lg:text-4xl font-bold"
            style={{ color: "var(--text-primary)" }}
          >
            {title}
          </h2>
          {seeMoreLink && (
            <Link
              href={seeMoreLink}
              className="text-sm sm:text-base font-medium hover:underline flex-shrink-0"
              style={{ color: "var(--text-primary)" }}
            >
              See More
            </Link>
          )}
        </div>
      )}

      <div className="relative">
        {/* Left Arrow - Hidden on mobile, shown on tablet+ */}
        {showLeftArrow && (
          <button
            onClick={() => scroll("left")}
            className="absolute left-0 top-1/2 -translate-y-1/2 z-10 bg-white rounded-full p-2 md:p-2.5 lg:p-3 border border-gray-200 shadow-sm hover:shadow-md transition-all duration-200 hidden md:block"
            aria-label="Scroll left"
          >
            <ChevronLeft className="w-4 h-4 md:w-5 md:h-5 lg:w-6 lg:h-6 text-gray-700" />
          </button>
        )}

        {/* Cards Container - Overflow hidden to clip partial cards */}
        <div className="overflow-hidden">
          <div
            ref={scrollRef}
            className="flex gap-3 sm:gap-4 overflow-x-auto scrollbar-hide scroll-smooth pb-4 items-stretch px-2 sm:px-0"
            style={{
              scrollbarWidth: "none",
              msOverflowStyle: "none",
              WebkitOverflowScrolling: "touch",
            }}
          >
            {children}
          </div>
        </div>

        {/* Right Arrow - Hidden on mobile, shown on tablet+ */}
        {showRightArrow && (
          <button
            onClick={() => scroll("right")}
            className="absolute right-0 top-1/2 -translate-y-1/2 z-10 bg-white rounded-full p-2 md:p-2.5 lg:p-3 border border-gray-200 shadow-sm hover:shadow-md transition-all duration-200 hidden md:block"
            aria-label="Scroll right"
          >
            <ChevronRight className="w-4 h-4 md:w-5 md:h-5 lg:w-6 lg:h-6 text-gray-700" />
          </button>
        )}
      </div>
    </div>
  );
}
