import { Container } from "@/components/layout/Container";
import { PageSection } from "@/components/layout/PageSection";
import { VenueCard } from "@/features/home/components/VenueCard";
import { EmptyState } from "@/components/data-display/EmptyState";
import { getHomeVenuesServer } from "@/features/home/api/getHomeVenues-server";
import { transformSalonToVenue } from "@/features/home/utils/transformSalonToVenue";
import { Venue } from "@/types/venue";

interface SaloonsPageProps {
  searchParams: Promise<{
    lat?: string;
    lng?: string;
    sort?: "near-you" | "top-rated" | "specialists";
  }>;
}

export default async function SaloonsPage({ searchParams }: SaloonsPageProps) {
  const params = await searchParams;

  // Get lat/lng from URL or use defaults
  const lat = params.lat ? parseFloat(params.lat) : 31.462576;
  const lng = params.lng ? parseFloat(params.lng) : 74.247932;
  const sortType = params.sort || "near-you";

  // Fetch venues data on the server
  const location = {
    targetLat: lat,
    targetLng: lng,
    rangeInKm: 500,
    serviceType: null,
  };

  let allVenues: Venue[] = [];
  let pageTitle = "Saloons Near You";

  try {
    const homeData = await getHomeVenuesServer(location);

    // Filter and sort based on sort type
    let filteredSalons = homeData?.slons ? [...homeData.slons] : [];

    if (sortType === "specialists") {
      filteredSalons = filteredSalons.filter(
        (salon) => salon.isExternalEmployee === true
      );
      pageTitle = "Specialists";
    } else if (sortType === "top-rated") {
      // Exclude external employees from top-rated
      filteredSalons = filteredSalons.filter(
        (salon) =>
          salon.isExternalEmployee === false &&
          salon.salonAverageRating !== null
      );
      pageTitle = "Top Rated Saloons";
    } else if (sortType === "near-you") {
      // Exclude external employees from near-you
      filteredSalons = filteredSalons.filter(
        (salon) => salon.isExternalEmployee === false
      );
      pageTitle = "Saloons Near You";
    }

    // Sort based on sort type
    let sortedSalons = filteredSalons;
    if (sortType === "near-you") {
      sortedSalons = filteredSalons.sort((a, b) => a.distance - b.distance);
    } else if (sortType === "top-rated") {
      sortedSalons = filteredSalons.sort((a, b) => {
        const ratingA = a.salonAverageRating ?? 0;
        const ratingB = b.salonAverageRating ?? 0;
        return ratingB - ratingA; // Descending order (highest first)
      });
    } else if (sortType === "specialists") {
      // For specialists, sort by distance
      sortedSalons = filteredSalons.sort((a, b) => a.distance - b.distance);
    }

    // Transform salon data to venue format
    allVenues = sortedSalons.map(transformSalonToVenue);
  } catch (error) {
    console.error("Failed to fetch saloons:", error);
    allVenues = [];
  }

  return (
    <main
      className="min-h-screen"
      style={{ backgroundColor: "var(--surface)" }}
    >
      <PageSection className="py-12">
        <Container>
          <h1 className="heading-2 mb-8">{pageTitle}</h1>

          {allVenues.length === 0 ? (
            <EmptyState
              title={pageTitle}
              message="No saloons found in your area. Check back later for new saloons near you!"
            />
          ) : (
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
              {allVenues.map((venue) => (
                <VenueCard key={venue.id} venue={venue} />
              ))}
            </div>
          )}
        </Container>
      </PageSection>
    </main>
  );
}
