import { HeroSection } from "@/features/home/components/HeroSection";
import { RecommendedSection } from "@/features/home/components/RecommendedSection";
import { NewToFreshaSection } from "@/features/home/components/NewToFreshaSection";
import { SpecialistsSection } from "@/features/home/components/SpecialistsSection";
import { AppDownloadSection } from "@/features/home/components/AppDownloadSection";
import { ReviewsSection } from "@/features/home/components/ReviewsSection";
import { TopStatsSection } from "@/features/home/components/TopStatsSection";
import { BusinessPromoSection } from "@/features/home/components/BusinessPromoSection";
import { CityDirectorySection } from "@/features/home/components/CityDirectorySection";
import { getHomeVenuesServer } from "@/features/home/api/getHomeVenues-server";
import { transformSalonToVenue } from "@/features/home/utils/transformSalonToVenue";

export default async function HomePage() {
  // Fetch venues data on the server
  // Using default location (Lahore, Pakistan) - can be enhanced with geolocation headers
  const defaultLocation = {
    targetLat: 31.462576,
    targetLng: 74.247932,
    rangeInKm: 10,
    serviceType: null,
  };

  let homeData;
  try {
    homeData = await getHomeVenuesServer(defaultLocation);
  } catch (error) {
    console.error("Failed to fetch home venues:", error);
    // If it's a Zod validation error, log the details
    if (error instanceof Error && error.name === "ZodError") {
      console.error("Zod validation error details:", error);
    }
    homeData = null;
  }

  // Sort and filter saloons before transforming
  // For "Near You" section: sort by distance (closest first), exclude external employees
  const nearYouSalons = homeData?.slons
    ? [...homeData.slons]
        .filter((salon) => salon.isExternalEmployee === false)
        .sort((a, b) => a.distance - b.distance)
    : [];
  const nearYouVenues = nearYouSalons.map(transformSalonToVenue);

  // For "Top rated" section: sort by rating (highest first), filter out null ratings and external employees
  const topRatedSalons = homeData?.slons
    ? [...homeData.slons]
        .filter((salon) => salon.isExternalEmployee === false && salon.salonAverageRating !== null)
        .sort((a, b) => {
          const ratingA = a.salonAverageRating ?? 0;
          const ratingB = b.salonAverageRating ?? 0;
          return ratingB - ratingA; // Descending order (highest first)
        })
    : [];
  const topRatedVenues = topRatedSalons.map(transformSalonToVenue);

  // For "Specialists" section: filter by isExternalEmployee = true
  const specialistsSalons = homeData?.slons
    ? homeData.slons.filter((salon) => salon.isExternalEmployee === true)
    : [];
  const specialistsVenues = specialistsSalons.map(transformSalonToVenue);

  return (
    <main
      className="min-h-screen overflow-x-hidden"
      style={{ backgroundColor: "var(--surface)" }}
    >
      <HeroSection />
      <RecommendedSection 
        venues={nearYouVenues} 
        lat={defaultLocation.targetLat}
        lng={defaultLocation.targetLng}
      />
      <NewToFreshaSection 
        venues={topRatedVenues}
        lat={defaultLocation.targetLat}
        lng={defaultLocation.targetLng}
      />
      <SpecialistsSection 
        venues={specialistsVenues}
        lat={defaultLocation.targetLat}
        lng={defaultLocation.targetLng}
      />
      <AppDownloadSection />
      <ReviewsSection />
      <TopStatsSection />
      <BusinessPromoSection />
      <CityDirectorySection />
    </main>
  );
}
