"use client";

import { useState, useEffect, Suspense } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import { ArrowLeft } from "lucide-react";
import { AuthLayoutSplit } from "@/features/auth/components/AuthLayoutSplit";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { AuthFooterLinks } from "@/features/auth/components/AuthFooterLinks";
import { verifyOtp } from "@/features/auth/api/verifyOtp";
import { resendOtp } from "@/features/auth/api/resendOtp";
import { setAccessToken, setUserData } from "@/lib/auth";
import { routes } from "@/config/routes";

function VerifyOtpPageContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const otpIdParam = searchParams.get("otpId");
  const userIdParam = searchParams.get("userId");
  const email = searchParams.get("email") || "";
  const isLoginFlow = searchParams.get("login") === "true";

  const [otp, setOtp] = useState("");
  const [error, setError] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [isResending, setIsResending] = useState(false);
  const [otpId, setOtpId] = useState<string | null>(otpIdParam || null);
  const [userId, setUserId] = useState<string | null>(userIdParam || null);
  const [otpSent, setOtpSent] = useState(false);

  // Auto-request OTP when coming from login without otpId
  useEffect(() => {
    if (!userIdParam) {
      router.push(isLoginFlow ? routes.login : routes.signup);
      return;
    }

    // If coming from login and no otpId, automatically request a new OTP
    const autoRequestOtp = async () => {
      if (isLoginFlow && !otpId && userId && email && !otpSent) {
        setIsLoading(true);
        setError("");

        try {
          const response = await resendOtp(email, userId.toString());

          if (response.status === "1" || response.status === "2") {
            setOtpId(response.data.otpId);
            setUserId(response.data.userId);
            setOtpSent(true);
          } else {
            setError(
              response.message ||
                "Failed to send verification code. Please try again."
            );
          }
        } catch (err: any) {
          console.error("Auto-resend OTP error:", err);
          setError(
            err.message || "Failed to send verification code. Please try again."
          );
        } finally {
          setIsLoading(false);
        }
      }
    };

    autoRequestOtp();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [userIdParam, isLoginFlow]);

  const handleOtpChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value.replace(/\D/g, "").slice(0, 6);
    setOtp(value);
    setError(""); // Clear error when user starts typing
  };

  const handleResendOtp = async (autoResend = false) => {
    if (!userId || !email) {
      setError("Unable to resend OTP. Please try logging in again.");
      return;
    }

    if (!autoResend) {
      setIsResending(true);
    } else {
      setIsLoading(true);
    }
    setError("");

    try {
      const response = await resendOtp(email, userId);

      if (response.status === "1" || response.status === "2") {
        // Successfully sent OTP
        setOtpId(response.data.otpId);
        setUserId(response.data.userId);
        setOtpSent(true);
        setError(""); // Clear any previous errors

        if (!autoResend) {
          // Show success message only if manually requested
          setTimeout(() => {
            setOtpSent(false);
          }, 3000);
        }
      } else {
        setError(
          response.message ||
            "Failed to send verification code. Please try again."
        );
      }
    } catch (err: any) {
      console.error("Resend OTP error:", err);
      setError(
        err.message || "Failed to send verification code. Please try again."
      );
    } finally {
      setIsResending(false);
      if (autoResend) {
        setIsLoading(false);
      }
    }
  };

  const handleVerify = async () => {
    if (!otp || otp.length < 4) {
      setError("Please enter a valid verification code");
      return;
    }

    if (!otpId || !userId) {
      setError(
        "Verification code not available. Please click 'Resend code' to receive a new one."
      );
      return;
    }

    setIsLoading(true);
    setError("");

    try {
      const userData = await verifyOtp(parseInt(otpId), parseInt(userId), otp);

      // Store authentication data
      setAccessToken(userData.accessToken);
      setUserData(userData);

      // Navigate to home page
      router.push(routes.home);
    } catch (err: any) {
      console.error("OTP verification error:", err);
      // Clear OTP input on error so user can re-enter
      setOtp("");

      // Show specific error message
      if (
        err.message?.toLowerCase().includes("invalid") ||
        err.message?.toLowerCase().includes("incorrect") ||
        err.message?.toLowerCase().includes("wrong")
      ) {
        setError(
          "The verification code you entered is incorrect. Please check your email and try again, or click 'Resend code' to receive a new one."
        );
      } else {
        setError(
          err.message ||
            "Verification failed. Please try again or click 'Resend code' to receive a new verification code."
        );
      }
    } finally {
      setIsLoading(false);
    }
  };

  if (!userId) {
    return null;
  }

  return (
    <AuthLayoutSplit>
      <div className="max-w-md mx-auto w-full flex flex-col min-h-full">
        <div className="mb-8">
          <button
            onClick={() => router.back()}
            className="mb-6 p-2 -ml-2 hover:bg-gray-50 rounded-lg transition-colors"
            aria-label="Go back"
          >
            <ArrowLeft className="w-5 h-5 text-gray-900" />
          </button>
          <h1 className="heading-2 mb-2">
            {isLoginFlow ? "Verify your account" : "Verify your email"}
          </h1>
          <p className="body-text-muted">
            {otpSent && !isLoginFlow
              ? "A new verification code has been sent to "
              : "We've sent a verification code to "}
            <span className="font-semibold text-primary-500">{email}</span>
            {otpSent && !isLoginFlow && (
              <span className="block mt-1 text-sm text-green-600">
                Code sent successfully!
              </span>
            )}
          </p>
        </div>

        <form
          onSubmit={(e) => {
            e.preventDefault();
            handleVerify();
          }}
          className="flex flex-col flex-1"
        >
          <div className="mb-6">
            <Input
              id="otp"
              label="Enter verification code"
              type="text"
              inputMode="numeric"
              value={otp}
              onChange={handleOtpChange}
              placeholder="000000"
              maxLength={6}
              error={error}
              touched={!!error}
              required
              className="text-center text-2xl tracking-widest font-semibold"
              disabled={isLoading || !otpId}
            />
            <p className="body-text-sm text-muted mt-2 text-center">
              Enter the 4-6 digit code sent to your email
            </p>
            {!otpId && (
              <p className="body-text-sm text-warning-500 mt-2 text-center">
                Requesting verification code...
              </p>
            )}
          </div>

          {error && (
            <div className="mb-4 p-3 rounded-xl bg-error-50 border border-error-200">
              <p className="body-text-sm text-error-500">{error}</p>
            </div>
          )}

          <Button
            type="submit"
            disabled={!otp || otp.length < 4 || !otpId || isLoading}
            loading={isLoading}
            className="w-full mb-auto"
          >
            Verify
          </Button>

          <div className="text-center mb-auto">
            <p className="body-text-sm text-muted mb-2">
              Didn't receive the code?
            </p>
            <button
              type="button"
              onClick={() => handleResendOtp(false)}
              disabled={isResending || isLoading}
              className="body-text-sm text-primary-500 hover:underline font-medium disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isResending ? "Sending..." : "Resend code"}
            </button>
          </div>

          <AuthFooterLinks />
        </form>
      </div>
    </AuthLayoutSplit>
  );
}

export default function VerifyOtpPage() {
  return (
    <Suspense fallback={
      <AuthLayoutSplit>
        <div className="max-w-md mx-auto w-full flex flex-col min-h-full items-center justify-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-gray-900"></div>
        </div>
      </AuthLayoutSplit>
    }>
      <VerifyOtpPageContent />
    </Suspense>
  );
}
