"use client";

import { useState } from "react";
import { useRouter } from "next/navigation";
import { AuthLayoutSplit } from "@/features/auth/components/AuthLayoutSplit";
import { AuthHeader } from "@/features/auth/components/AuthHeader";
import { SocialLoginButton } from "@/features/auth/components/SocialLoginButton";
import { OrDivider } from "@/features/auth/components/OrDivider";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { AuthFooterLinks } from "@/features/auth/components/AuthFooterLinks";
import { routes } from "@/config/routes";
import { checkEmail } from "@/features/auth/api/emailChecker";
import { setAccessToken, setUserData } from "@/lib/auth";
import { handleSocialAuth } from "@/features/auth/api/socialAuth";
import Link from "next/link";

export default function SignupPage() {
  const router = useRouter();
  const [email, setEmail] = useState("");
  const [emailError, setEmailError] = useState("");
  const [touched, setTouched] = useState(false);
  const [isLoading, setIsLoading] = useState(false);

  const validateEmail = (emailValue: string): string => {
    if (!emailValue || emailValue.trim().length === 0) {
      return "Email address is required";
    }
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(emailValue.trim())) {
      return "Please enter a valid email address";
    }
    return "";
  };

  const handleEmailChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value;
    setEmail(value);
    if (touched) {
      setEmailError(validateEmail(value));
    }
  };

  const handleEmailBlur = () => {
    setTouched(true);
    setEmailError(validateEmail(email));
  };

  const handleContinue = async () => {
    setTouched(true);
    const error = validateEmail(email);
    setEmailError(error);
    
    if (error || !email) {
      return;
    }

    setIsLoading(true);
    try {
      // Check if email exists
      const response = await checkEmail(email);
      
      // Check if user exists (status "1" or "2" with user data)
      if (response.status === "1" || response.status === "2") {
        const userData = response.data as any;
        
        // If user data exists, user is already registered - log them in
        if (userData && userData.accessToken) {
          setAccessToken(userData.accessToken);
          setUserData(userData);
          router.push(routes.home);
          return;
        }
      }
      
      // User doesn't exist - navigate to signup form
      router.push(`/signup/create?email=${encodeURIComponent(email)}`);
    } catch (err: any) {
      // If error, assume user doesn't exist and navigate to signup
      console.error("Email check error:", err);
      router.push(`/signup/create?email=${encodeURIComponent(email)}`);
    } finally {
      setIsLoading(false);
    }
  };

  const handleSocialLogin = async (provider: "google" | "apple") => {
    setIsLoading(true);
    try {
      const result = await handleSocialAuth(provider);
      
      if (result.success) {
        if (result.requiresSignup && result.userData) {
          // Data is incomplete - store in sessionStorage and redirect to signup form
          const { setSocialAuthData } = await import("@/lib/socialAuthStorage");
          setSocialAuthData(result.userData);
          router.push(`${routes.signupCreate}?socialAuth=true`);
        } else if (result.requiresOtp) {
          // If OTP is required, redirect to OTP page
          router.push(routes.signupVerifyOtp);
        } else {
          // User is logged in, redirect to home
          router.push(routes.home);
        }
      } else {
        setEmailError(result.error || "Authentication failed. Please try again.");
      }
    } catch (error: any) {
      console.error("Social login error:", error);
      setEmailError(error.message || "Authentication failed. Please try again.");
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <AuthLayoutSplit>
      <div className="max-w-md mx-auto w-full flex flex-col min-h-full">
        <AuthHeader
          title="Fresha for customers"
          subtitle="Create an account or log in to book and manage your appointments."
        />

        {/* Social Login Buttons */}
        <div className="space-y-3 mb-6">
          <SocialLoginButton
            provider="google"
            onClick={() => handleSocialLogin("google")}
            loading={isLoading}
          >
            Continue with Google
          </SocialLoginButton>
          <SocialLoginButton
            provider="apple"
            onClick={() => handleSocialLogin("apple")}
            loading={isLoading}
          >
            Continue with Apple
          </SocialLoginButton>
        </div>

        <OrDivider />

        {/* Email Input */}
        <div className="mb-6">
          <Input
            id="email"
            type="email"
            placeholder="Email address"
            value={email}
            onChange={handleEmailChange}
            onBlur={handleEmailBlur}
            error={emailError}
            touched={touched}
            onKeyDown={(e) => {
              if (e.key === "Enter" && email && !emailError) {
                handleContinue();
              }
            }}
          />
        </div>

        {/* Continue Button */}
        <Button
          onClick={handleContinue}
          disabled={!email || !!emailError}
          loading={isLoading}
          className="w-full mb-6"
        >
          Continue
        </Button>

        {/* Business Account Link */}
        <p className="body-text text-center mb-auto">
          Have a business account?{" "}
          <Link
            href="/login/professional"
            className="text-primary-500 hover:underline font-medium"
          >
            Sign in as a professional
          </Link>
        </p>

        {/* Footer Links */}
        <AuthFooterLinks />
      </div>
    </AuthLayoutSplit>
  );
}
