# Theme System Guide

## Overview

This project uses a centralized theme system that allows you to change colors, fonts, spacing, and other design tokens from a single file: `src/config/theme.ts`.

## File Structure

```
src/
├── config/
│   └── theme.ts          # Main theme configuration (SINGLE SOURCE OF TRUTH)
├── styles/
│   ├── themes.css        # CSS custom properties generated from theme.ts
│   ├── typography.css    # Typography utilities
│   └── globals.css       # Global styles that import theme files
└── tailwind.config.ts    # Tailwind config that extends theme values
```

## How to Change Theme Values

### 1. Edit `src/config/theme.ts`

This is the **ONLY** file you need to edit to change theme values. For example:

```typescript
// Change primary brand color
colors: {
  primary: {
    500: "#9333ea",  // Change this to your new brand color
  },
}

// Change font family
typography: {
  fontFamily: {
    sans: ["Your Font", "system-ui", "sans-serif"],
  },
}

// Change spacing scale
spacing: {
  4: "1rem",  // Change base spacing unit
}
```

### 2. CSS Variables Auto-Update

The CSS variables in `src/styles/themes.css` are manually maintained to match `theme.ts`. When you change `theme.ts`, update the corresponding CSS variables in `themes.css`.

### 3. Tailwind Classes Auto-Update

Tailwind config automatically uses theme values, so classes like `bg-primary-500`, `text-gray-900`, etc. will use your theme values.

## Using Theme Values

### In Tailwind Classes (Recommended)

```tsx
// Colors
<div className="bg-primary-500 text-white">
<div className="bg-gray-50 text-gray-900">
<div className="border-gray-200">

// Typography
<h1 className="text-3xl font-bold text-gray-900">
<p className="text-sm text-gray-600 leading-snug">

// Spacing
<div className="p-5 mb-4 gap-2">

// Shadows
<div className="shadow-lg hover:shadow-xl">

// Border Radius
<button className="rounded-2xl">
```

### In CSS/Inline Styles

```css
/* Using CSS variables */
.my-component {
  background-color: var(--color-primary-500);
  color: var(--color-gray-900);
  box-shadow: var(--shadow-lg);
}
```

### In TypeScript/JavaScript

```typescript
import { theme } from "@/config/theme";

const primaryColor = theme.colors.primary[500];
const baseSpacing = theme.spacing[4];
```

## Theme Structure

### Colors

- **Primary**: Brand purple colors (`primary-50` to `primary-950`)
- **Secondary**: Brand pink colors (`secondary-50` to `secondary-950`)
- **Gray**: Neutral grays (`gray-50` to `gray-950`)
- **Semantic**: Success, Warning, Error, Info colors
- **Special**: Yellow (for stars), White, Black

**Usage:**
```tsx
className="bg-primary-500 text-white"
className="text-gray-900 bg-gray-50"
className="border-gray-200"
```

### Typography

- **Font Families**: `font-sans`, `font-mono`
- **Font Sizes**: `text-xs`, `text-sm`, `text-base`, `text-lg`, `text-xl`, `text-2xl`, etc.
- **Font Weights**: `font-light`, `font-normal`, `font-medium`, `font-semibold`, `font-bold`
- **Line Heights**: `leading-tight`, `leading-snug`, `leading-normal`, `leading-relaxed`

**Usage:**
```tsx
<h1 className="text-3xl font-bold text-gray-900 leading-tight">
<p className="text-sm text-gray-600 leading-snug">
```

### Spacing

Scale: `0`, `1` (4px), `2` (8px), `3` (12px), `4` (16px), `5` (20px), `6` (24px), `8` (32px), etc.

**Usage:**
```tsx
<div className="p-5 mb-4 gap-2">
```

### Shadows

- `shadow-sm`, `shadow` (base), `shadow-md`, `shadow-lg`, `shadow-xl`, `shadow-2xl`

**Usage:**
```tsx
<div className="shadow-lg hover:shadow-xl">
```

### Border Radius

- `rounded-sm`, `rounded`, `rounded-md`, `rounded-lg`, `rounded-xl`, `rounded-2xl`, `rounded-3xl`, `rounded-full`

**Usage:**
```tsx
<button className="rounded-2xl">
```

## Common Patterns

### Cards
```tsx
<div className="bg-white rounded-2xl shadow-sm p-5">
```

### Buttons
```tsx
<button className="bg-primary-500 text-white rounded-lg px-4 py-2 font-medium hover:bg-primary-600 transition-colors">
```

### Text Hierarchy
```tsx
<h1 className="text-3xl font-bold text-gray-900">Title</h1>
<p className="text-base text-gray-600">Body text</p>
<p className="text-sm text-gray-500">Small text</p>
```

### Borders
```tsx
<div className="border border-gray-200 rounded-lg">
```

## Migration Checklist

When updating components to use theme:

- [ ] Replace hardcoded hex colors with theme color classes
- [ ] Replace hardcoded font names with theme font classes
- [ ] Replace hardcoded spacing values with theme spacing scale
- [ ] Replace hardcoded shadows with theme shadow classes
- [ ] Replace hardcoded border radius with theme radius classes

## Example: Before & After

### Before (Hardcoded)
```tsx
<div className="bg-[#9333ea] text-[#ffffff] p-[20px] rounded-[16px] shadow-[0_1px_3px_rgba(0,0,0,0.1)]">
```

### After (Theme-based)
```tsx
<div className="bg-primary-500 text-white p-5 rounded-2xl shadow">
```

## Benefits

1. **Single Source of Truth**: Change `theme.ts` to update entire app
2. **Type Safety**: TypeScript ensures correct usage
3. **Consistency**: All components use same design tokens
4. **Maintainability**: Easy to update brand colors/fonts
5. **Future-Proof**: Easy to add dark mode or theme switching

## Next Steps

1. Update existing components to use theme classes
2. Remove all hardcoded colors, fonts, and spacing
3. Test theme changes by updating `theme.ts`
4. Document any custom theme extensions




