# Styling System Guide

## Core Principle

**Never define typography properties individually.** Always use semantic classes that combine font-size, font-weight, line-height, and color.

## Typography Classes

### Headings
- `.heading-1` through `.heading-6` - Standard headings
- `.heading-hero` - Responsive hero heading (scales 48px → 60px → 72px)

### Body Text
- `.body-text` - Regular body text (16px)
- `.body-text-lg` - Larger body text (18px)
- `.body-text-sm` - Smaller body text (14px)
- `.body-text-muted` - Muted body text variants

### Card Text
- `.card-title` - Card/venue names (16px, semibold)
- `.card-text` - Card descriptions/addresses (14px, muted)
- `.card-category` - Categories/tags (12px, muted)

### Special Cases
- `.stat-number` - Statistics numbers (responsive)
- `.stat-number-large` - Large statistics (like "1 billion+")
- `.review-title` - Review titles (18px, semibold)
- `.review-body` - Review body text (16px, muted)
- `.reviewer-name` - Reviewer names (16px, medium)
- `.reviewer-location` - Reviewer locations (14px, muted)

### Buttons & Navigation
- `.button-text` - Standard button text
- `.button-text-sm` - Small button text
- `.button-text-lg` - Large button text
- `.nav-link` - Navigation links

## Color Classes

Use these instead of inline `style={{ color: "var(--text-primary)" }}`:

- `.text-primary` - Primary text color
- `.text-secondary` - Secondary text color
- `.text-muted` - Muted text color
- `.bg-surface` - Surface background
- `.bg-primary-500` - Primary background

## Before & After Examples

### ❌ Don't Do This:
```tsx
<h1 className="text-3xl md:text-4xl font-bold text-gray-900 mb-8" style={{ color: "var(--text-primary)" }}>
  Title
</h1>
<p className="text-lg text-gray-600" style={{ color: "var(--text-muted)" }}>
  Description
</p>
```

### ✅ Do This:
```tsx
<h1 className="heading-2 mb-8">
  Title
</h1>
<p className="body-text-muted-lg">
  Description
</p>
```

### ❌ Don't Do This:
```tsx
<h3 className="text-base font-semibold leading-tight" style={{ color: "var(--text-primary)" }}>
  Card Title
</h3>
<p className="text-sm mb-1.5 leading-snug" style={{ color: "var(--text-muted)" }}>
  Address
</p>
```

### ✅ Do This:
```tsx
<h3 className="card-title">
  Card Title
</h3>
<p className="card-text mb-1.5">
  Address
</p>
```

## Dynamic Styles

For truly dynamic styles (like conditional colors based on state), minimal inline styles are acceptable:

```tsx
// ✅ Acceptable - Dynamic state-based color
<button
  className="px-4 py-2 rounded-full label"
  style={{
    backgroundColor: active ? "var(--text-primary)" : "transparent",
    color: active ? "var(--surface)" : "var(--text-primary)",
  }}
>
  {country}
</button>
```

But still use semantic typography classes for the text itself.

## Rules

1. **Always use semantic typography classes** - Never define font-size, font-weight, line-height individually
2. **Use color utility classes** - Use `.text-primary`, `.text-muted` instead of inline color styles
3. **Combine with spacing utilities** - Typography classes handle text, spacing classes handle layout
4. **Minimize inline styles** - Only use for truly dynamic values
5. **Refer to TYPOGRAPHY_GUIDE.md** - Complete reference for all available classes




