/**
 * Social Auth Data Storage
 * Temporarily stores social auth data in sessionStorage
 * to avoid passing sensitive data in URL parameters
 */

const SOCIAL_AUTH_DATA_KEY = "fresha_social_auth_data";

export interface SocialAuthData {
  email: string;
  firstName: string;
  lastName: string;
  phoneNumber: string;
  image: string;
  signedBy: "google" | "apple";
}

/**
 * Store social auth data
 */
export function setSocialAuthData(data: SocialAuthData): void {
  if (typeof window !== "undefined") {
    sessionStorage.setItem(SOCIAL_AUTH_DATA_KEY, JSON.stringify(data));
  }
}

/**
 * Get social auth data
 */
export function getSocialAuthData(): SocialAuthData | null {
  if (typeof window === "undefined") {
    return null;
  }
  const data = sessionStorage.getItem(SOCIAL_AUTH_DATA_KEY);
  return data ? JSON.parse(data) : null;
}

/**
 * Clear social auth data
 */
export function clearSocialAuthData(): void {
  if (typeof window !== "undefined") {
    sessionStorage.removeItem(SOCIAL_AUTH_DATA_KEY);
  }
}




