/**
 * Zod Schemas for Venue entities
 */

import { z } from "zod";

export const OpeningHoursSchema = z.record(
  z.string(),
  z
    .object({
      open: z.string(),
      close: z.string(),
    })
    .nullable()
);

export const VenueSchema = z.object({
  id: z.string(),
  slug: z.string(),
  name: z.string(),
  image: z.string().refine(
    (val) => {
      // Allow valid URLs, data URLs, or placeholder marker
      return (
        val.startsWith("http://") ||
        val.startsWith("https://") ||
        val.startsWith("data:") ||
        val === "/placeholder-venue"
      );
    },
    { message: "Image must be a valid URL, data URL, or placeholder" }
  ),
  rating: z.number().min(0).max(5),
  reviewCount: z.number().int().min(0),
  category: z.string(),
  location: z.string(),
  address: z.string(),
  description: z.string().optional(),
  openingHours: OpeningHoursSchema.optional(),
  badge: z.string().optional(),
  phoneNumber: z.string().optional(),
  website: z.string().url().optional(),
  latitude: z.number().optional(),
  longitude: z.number().optional(),
  createdAt: z.string().datetime().optional(),
  updatedAt: z.string().datetime().optional(),
});

export type Venue = z.infer<typeof VenueSchema>;
export type OpeningHours = z.infer<typeof OpeningHoursSchema>;




