/**
 * Zod Schemas for Booking entities
 */

import { z } from "zod";

export const TimeSlotSchema = z.object({
  startTime: z.string().datetime(),
  endTime: z.string().datetime(),
  available: z.boolean(),
  professionalId: z.string().optional(),
  professionalName: z.string().optional(),
});

export const BookingServiceSchema = z.object({
  serviceId: z.string(),
  serviceName: z.string(),
  duration: z.number().int(),
  price: z.number().min(0),
  currency: z.string().default("USD"),
});

export const BookingSchema = z.object({
  id: z.string(),
  userId: z.string(),
  venueId: z.string(),
  venueName: z.string(),
  services: z.array(BookingServiceSchema),
  professionalId: z.string().optional(),
  professionalName: z.string().optional(),
  startTime: z.string().datetime(),
  endTime: z.string().datetime(),
  status: z.enum([
    "pending",
    "confirmed",
    "completed",
    "cancelled",
    "no_show",
  ]),
  totalPrice: z.number().min(0),
  currency: z.string().default("USD"),
  paymentStatus: z.enum(["pending", "paid", "refunded"]).default("pending"),
  notes: z.string().optional(),
  createdAt: z.string().datetime(),
  updatedAt: z.string().datetime().optional(),
});

export const CreateBookingRequestSchema = z.object({
  venueId: z.string(),
  services: z.array(z.string()).min(1),
  professionalId: z.string().optional(),
  startTime: z.string().datetime(),
  notes: z.string().optional(),
  discountCode: z.string().optional(),
});

export const AvailableSlotsRequestSchema = z.object({
  venueId: z.string(),
  serviceIds: z.array(z.string()).min(1),
  professionalId: z.string().optional(),
  date: z.string().date(),
  duration: z.number().int().min(1),
});

export type Booking = z.infer<typeof BookingSchema>;
export type BookingService = z.infer<typeof BookingServiceSchema>;
export type TimeSlot = z.infer<typeof TimeSlotSchema>;
export type CreateBookingRequest = z.infer<typeof CreateBookingRequestSchema>;
export type AvailableSlotsRequest = z.infer<typeof AvailableSlotsRequestSchema>;




