/**
 * Zod Schemas for Appointment Detail API
 */

import { z } from "zod";

export const AppointmentDetailAddressDBSchema = z
  .object({
    id: z.number().optional(),
    streetAddress: z.string(),
    building: z.string().optional(),
    floor: z.string().optional(),
    apartment: z.string().optional(),
    district: z.string().optional(),
    city: z.string(),
    country: z.string().optional(),
    lat: z.string().optional(),
    lng: z.string().optional(),
    postalCode: z.string().optional(),
    province: z.string().optional(),
  })
  .passthrough(); // Allow additional fields

export const AppointmentDetailServiceSchema = z.object({
  id: z.number(),
  serviceName: z.string(),
  price: z.string(),
  duration: z.number().optional(),
  professionalName: z.string().optional(),
});

export const AppointmentDetailSalonDetailSchema = z
  .object({
    id: z.number().optional(),
    salonName: z.string(),
    coverImage: z.string().nullable(),
    connectAccountId: z.string().optional(),
    isOnline: z.boolean().optional(),
    description: z.string().nullable().optional(),
    addressDB: AppointmentDetailAddressDBSchema.optional(),
    user: z.any().optional(), // User object structure
  })
  .passthrough(); // Allow additional fields

export const SalonPolicySchema = z.object({
  cancellationPolicy: z.string().optional(),
  reschedulePolicy: z.string().optional(),
  noShowPolicy: z.string().optional(),
  policyVersion: z.string().optional(),
});

export const AppointmentDetailSchema = z
  .object({
    id: z.number(),
    status: z.string(), // "book", "complete", etc.
    startTime: z.string(), // "14:01:00"
    endTime: z.string(), // "14:36:00"
    on: z.string(), // "2025-10-15"
    total: z.string(),
    subTotal: z.string(),
    noshowCharges: z.string(),
    cancellationCharges: z.string(),
    serviceCount: z.number().optional(),
    salonDetail: AppointmentDetailSalonDetailSchema,
    services: z.array(AppointmentDetailServiceSchema).optional(),
    jobs: z.array(z.any()).optional(), // API returns "jobs" instead of "services"
    salonPolicy: SalonPolicySchema.optional(),
    paidUpfront: z.string().optional(),
    amountToPayAtVenue: z.string().optional(),
    bookingReference: z.string().optional(),
    // Additional fields that may be present
    discount: z.string().optional(),
    actualCapturedAmount: z.string().optional(),
    platformFee: z.string().optional(),
    tip: z.string().optional(),
    initialPayment: z.string().optional(),
    duration: z.number().optional(),
    rating: z.string().optional(),
    upFrontPaymentPercentage: z.number().nullable().optional(),
    initialPaymentIntend: z.string().nullable().optional(),
    cancellationPolicyVersion: z.number().optional(),
    reschedulePolicyVersion: z.number().optional(),
    upFrontPolicyVersion: z.number().optional(),
    noshowPolicyVersion: z.number().optional(),
    penaltyAmount: z.string().optional(),
    refundAmount: z.string().optional(),
    reScheduleCount: z.number().optional(),
    finalPayment: z.string().optional(),
    paymentMethod: z.string().optional(),
    feedback: z.string().nullable().optional(),
    salonReply: z.string().nullable().optional(),
    vatAmount: z.string().nullable().optional(),
    vatOriginalAmount: z.string().nullable().optional(),
    appointmentRating: z.string().nullable().optional(),
    couponOnBookings: z.array(z.any()).optional(),
  })
  .passthrough(); // Allow additional fields

// API response structure: { appointments: {...}, policies: {...}, currencyUnit: "...", subscriptionExpire: boolean }
export const GetAppointmentDetailResponseSchema = z
  .object({
    appointments: AppointmentDetailSchema,
    policies: z.any().optional(), // Policies object structure
    currencyUnit: z.string(),
    subscriptionExpire: z.boolean().optional(),
  })
  .transform((data) => {
    // Normalize to always have appointment key (singular) for consistency
    return {
      appointment: data.appointments,
      policies: data.policies,
      currencyUnit: data.currencyUnit,
    };
  });

export type AppointmentDetail = z.infer<typeof AppointmentDetailSchema>;
export type GetAppointmentDetailResponse = z.infer<
  typeof GetAppointmentDetailResponseSchema
>;
