/**
 * Device Token Management
 * Generates and stores a unique device token for the current browser/device
 */

const DEVICE_TOKEN_KEY = "fresha_dv_token";

/**
 * Generate a unique device token
 */
function generateDeviceToken(): string {
  // Generate a unique token based on browser fingerprint
  const timestamp = Date.now();
  const random = Math.random().toString(36).substring(2, 15);
  const userAgent = typeof window !== "undefined" ? window.navigator.userAgent : "";
  const screen = typeof window !== "undefined" 
    ? `${window.screen.width}x${window.screen.height}` 
    : "";
  
  const fingerprint = `${timestamp}-${random}-${userAgent}-${screen}`;
  
  // Create a simple hash
  let hash = 0;
  for (let i = 0; i < fingerprint.length; i++) {
    const char = fingerprint.charCodeAt(i);
    hash = ((hash << 5) - hash) + char;
    hash = hash & hash; // Convert to 32-bit integer
  }
  
  return `${hash}${timestamp}${random}`;
}

/**
 * Get or create device token
 * Stores token in localStorage for persistence
 */
export function getDeviceToken(): string {
  if (typeof window === "undefined") {
    // Server-side: return a temporary token
    return "server-token";
  }

  let token = localStorage.getItem(DEVICE_TOKEN_KEY);

  if (!token) {
    token = generateDeviceToken();
    localStorage.setItem(DEVICE_TOKEN_KEY, token);
  }

  return token;
}

/**
 * Clear device token (for logout)
 */
export function clearDeviceToken(): void {
  if (typeof window !== "undefined") {
    localStorage.removeItem(DEVICE_TOKEN_KEY);
  }
}




