/**
 * Authentication Utilities
 * Handles token storage and authentication state
 */

const ACCESS_TOKEN_KEY = "fresha_access_token";
const USER_DATA_KEY = "fresha_user_data";

export interface UserData {
  accessToken: string;
  userId: string;
  image: string;
  firstName: string;
  lastName: string;
  email: string;
  phoneNum: string;
  stripeCustomerId: string | null;
  joinOn?: string;
}

/**
 * Store access token
 */
export function setAccessToken(token: string): void {
  if (typeof window !== "undefined") {
    localStorage.setItem(ACCESS_TOKEN_KEY, token);
    // Dispatch custom event to notify components of auth state change
    window.dispatchEvent(new CustomEvent("authStateChanged"));
  }
}

/**
 * Get access token
 */
export function getAccessToken(): string | null {
  if (typeof window === "undefined") {
    return null;
  }
  return localStorage.getItem(ACCESS_TOKEN_KEY);
}

/**
 * Store user data
 */
export function setUserData(userData: UserData): void {
  if (typeof window !== "undefined") {
    localStorage.setItem(USER_DATA_KEY, JSON.stringify(userData));
    // Dispatch custom event to notify components of auth state change
    window.dispatchEvent(new CustomEvent("authStateChanged"));
  }
}

/**
 * Get user data
 */
export function getUserData(): UserData | null {
  if (typeof window === "undefined") {
    return null;
  }
  const data = localStorage.getItem(USER_DATA_KEY);
  return data ? JSON.parse(data) : null;
}

/**
 * Check if user is authenticated
 */
export function isAuthenticated(): boolean {
  return !!getAccessToken();
}

/**
 * Clear authentication data (logout)
 */
export function clearAuth(): void {
  if (typeof window !== "undefined") {
    localStorage.removeItem(ACCESS_TOKEN_KEY);
    localStorage.removeItem(USER_DATA_KEY);
    // Dispatch custom event to notify components of auth state change
    window.dispatchEvent(new CustomEvent("authStateChanged"));
  }
}
