/**
 * Server-side API Fetcher
 * ONLY use in Server Components, Server Actions, Route Handlers
 */

import "server-only";
import { API_BASE_URL } from "@/config/env";
import { cookies } from "next/headers";

export interface ApiResponse<T = any> {
  status: string;
  message: string;
  data: T;
  error: string;
}

export class ApiError extends Error {
  constructor(
    public status: string,
    public message: string,
    public statusCode?: number,
    public error?: string
  ) {
    super(message);
    this.name = "ApiError";
  }
}

export interface FetcherOptions extends RequestInit {
  requireAuth?: boolean;
  headers?: HeadersInit;
}

/**
 * Normalize API response data to camelCase
 */
function normalizeToCamelCase(obj: any): any {
  if (obj === null || obj === undefined) {
    return obj;
  }

  if (Array.isArray(obj)) {
    return obj.map(normalizeToCamelCase);
  }

  if (typeof obj !== "object") {
    return obj;
  }

  const normalized: any = {};
  for (const [key, value] of Object.entries(obj)) {
    const camelKey = key.replace(/_([a-z])/g, (_, letter) => letter.toUpperCase());
    normalized[camelKey] = normalizeToCamelCase(value);
  }

  return normalized;
}

/**
 * Server-side fetcher
 * Reads tokens from cookies
 */
export async function serverFetcher<T = any>(
  endpoint: string,
  options: FetcherOptions = {}
): Promise<T> {
  const {
    requireAuth = true,
    headers: customHeaders = {},
    ...fetchOptions
  } = options;

  const url = `${API_BASE_URL}${endpoint}`;

  // Build headers
  const headersObj: Record<string, string> = {
    "Content-Type": "application/json",
    Accept: "application/json",
  };

  // Add custom headers if provided as a plain object
  if (customHeaders && typeof customHeaders === "object" && !Array.isArray(customHeaders) && !(customHeaders instanceof Headers)) {
    Object.assign(headersObj, customHeaders);
  }

  // Add authentication token from cookies
  if (requireAuth) {
    const cookieStore = await cookies();
    const token = cookieStore.get("access_token")?.value || null;
    
    if (token) {
      headersObj.accessToken = token;
    }
  }

  // Convert to HeadersInit for fetch
  const headers: HeadersInit = headersObj;

  // Add device token (skip on server - device tokens are client-only)
  // Device tokens are generated per device/browser, not per server request
  // If needed, generate a server-side session token instead

  try {
    console.log("[serverFetcher] Making request to:", url);
    console.log("[serverFetcher] Headers:", JSON.stringify(headers, null, 2));
    console.log("[serverFetcher] Method:", fetchOptions.method || "GET");
    
    const response = await fetch(url, {
      ...fetchOptions,
      headers,
    });

    console.log("[serverFetcher] Response status:", response.status);
    console.log("[serverFetcher] Response statusText:", response.statusText);
    console.log("[serverFetcher] Response headers:", Object.fromEntries(response.headers.entries()));

    // Handle non-JSON responses
    const contentType = response.headers.get("content-type");
    console.log("[serverFetcher] Content-Type:", contentType);
    
    if (!contentType?.includes("application/json")) {
      const text = await response.text();
      console.error("[serverFetcher] Non-JSON response body:", text.substring(0, 500));
      throw new ApiError(
        "0",
        `Unexpected response type: ${contentType}`,
        response.status
      );
    }

    const data: ApiResponse<T> = await response.json();
    console.log("[serverFetcher] Response data status:", data.status);
    console.log("[serverFetcher] Response data message:", data.message);
    console.log("[serverFetcher] Response has data:", !!data.data);

    // Check if status indicates an error
    if (data.status !== "1" && data.status !== "2") {
      console.error("[serverFetcher] API returned error status:", data.status);
      console.error("[serverFetcher] Error message:", data.message);
      console.error("[serverFetcher] Error field:", data.error);
      throw new ApiError(
        data.status,
        data.message,
        response.status,
        data.error
      );
    }

    // Normalize and return data
    console.log("[serverFetcher] Normalizing response data");
    const normalizedData = normalizeToCamelCase(data.data) as T;
    console.log("[serverFetcher] Normalized data keys:", normalizedData && typeof normalizedData === "object" ? Object.keys(normalizedData) : "N/A");
    return normalizedData;
  } catch (error) {
    // Re-throw ApiError as-is
    if (error instanceof ApiError) {
      throw error;
    }

    // Handle network errors
    if (error instanceof TypeError && error.message === "Failed to fetch") {
      throw new ApiError(
        "0",
        "Network error. Please check your connection.",
        0
      );
    }

    // Handle other errors
    throw new ApiError(
      "0",
      error instanceof Error ? error.message : "An unexpected error occurred",
      500
    );
  }
}

