/**
 * Get Venue Services API
 * Fetches all services for a venue
 */

import { clientFetcher } from "@/lib/api/fetcher-client";
import { ServiceSchema, type Service } from "@/lib/schemas/service";
import { z } from "zod";

const ServicesResponseSchema = z.array(ServiceSchema);

export interface GetVenueServicesParams {
  venueId: string;
  categoryId?: string;
}

/**
 * Server-side: Get venue services
 * Dynamically imports server fetcher to avoid bundling in client code
 */
export async function getVenueServicesServer(
  params: GetVenueServicesParams
): Promise<Service[]> {
  // Dynamically import server fetcher to avoid bundling in client
  const { serverFetcher } = await import("@/lib/api/fetcher-server");

  const queryParams = new URLSearchParams();
  if (params.categoryId) {
    queryParams.set("categoryId", params.categoryId);
  }

  const endpoint = `/venues/${params.venueId}/services${
    queryParams.toString() ? `?${queryParams.toString()}` : ""
  }`;

  const data = await serverFetcher<unknown>(endpoint, {
    requireAuth: false,
    method: "GET",
  });

  return ServicesResponseSchema.parse(data);
}

/**
 * Client-side: Get venue services
 */
export async function getVenueServicesClient(
  params: GetVenueServicesParams
): Promise<Service[]> {
  const queryParams = new URLSearchParams();
  if (params.categoryId) {
    queryParams.set("categoryId", params.categoryId);
  }

  const endpoint = `/venues/${params.venueId}/services${
    queryParams.toString() ? `?${queryParams.toString()}` : ""
  }`;

  const data = await clientFetcher<unknown>(endpoint, {
    requireAuth: false,
    method: "GET",
  });

  return ServicesResponseSchema.parse(data);
}
