/**
 * Get Venue Reviews API
 * Fetches reviews for a venue
 */

import { clientFetcher } from "@/lib/api/fetcher-client";
import { ReviewSchema, type Review } from "@/lib/schemas/review";
import { z } from "zod";

const ReviewsResponseSchema = z.object({
  reviews: z.array(ReviewSchema),
  total: z.number().int(),
  page: z.number().int().optional(),
  limit: z.number().int().optional(),
});

export interface GetVenueReviewsParams {
  venueId: string;
  page?: number;
  limit?: number;
  sortBy?: "newest" | "oldest" | "highest" | "lowest";
}

/**
 * Server-side: Get venue reviews
 * Dynamically imports server fetcher to avoid bundling in client code
 */
export async function getVenueReviewsServer(
  params: GetVenueReviewsParams
): Promise<{ reviews: Review[]; total: number }> {
  // Dynamically import server fetcher to avoid bundling in client
  const { serverFetcher } = await import("@/lib/api/fetcher-server");

  const queryParams = new URLSearchParams();
  if (params.page) queryParams.set("page", params.page.toString());
  if (params.limit) queryParams.set("limit", params.limit.toString());
  if (params.sortBy) queryParams.set("sortBy", params.sortBy);

  const endpoint = `/venues/${params.venueId}/reviews${
    queryParams.toString() ? `?${queryParams.toString()}` : ""
  }`;

  const data = await serverFetcher<unknown>(endpoint, {
    requireAuth: false,
    method: "GET",
  });

  const parsed = ReviewsResponseSchema.parse(data);
  return {
    reviews: parsed.reviews,
    total: parsed.total,
  };
}

/**
 * Client-side: Get venue reviews
 */
export async function getVenueReviewsClient(
  params: GetVenueReviewsParams
): Promise<{ reviews: Review[]; total: number }> {
  const queryParams = new URLSearchParams();
  if (params.page) queryParams.set("page", params.page.toString());
  if (params.limit) queryParams.set("limit", params.limit.toString());
  if (params.sortBy) queryParams.set("sortBy", params.sortBy);

  const endpoint = `/venues/${params.venueId}/reviews${
    queryParams.toString() ? `?${queryParams.toString()}` : ""
  }`;

  const data = await clientFetcher<unknown>(endpoint, {
    requireAuth: false,
    method: "GET",
  });

  const parsed = ReviewsResponseSchema.parse(data);
  return {
    reviews: parsed.reviews,
    total: parsed.total,
  };
}
