"use client";

import { useState, useEffect, useRef } from "react";
import { Search, MapPin } from "lucide-react";
import { getHomeVenuesClient } from "@/features/home/api/getHomeVenues-client";
import { transformSalonToVenue } from "@/features/home/utils/transformSalonToVenue";
import { Venue } from "@/types/venue";
import Image from "next/image";
import { API_BASE_URL } from "@/config/env";
import Link from "next/link";
import { routes } from "@/config/routes";
import { getServiceTypesClient } from "../api/getServiceTypes-client";

interface SearchSuggestionsProps {
  isOpen: boolean;
  onClose: () => void;
  searchQuery: string;
  location: { lat: number; lng: number } | null;
}

interface TreatmentSuggestion {
  name: string;
  type: "treatment";
}

export function SearchSuggestions({
  isOpen,
  onClose,
  searchQuery,
  location,
}: SearchSuggestionsProps) {
  const [treatments, setTreatments] = useState<TreatmentSuggestion[]>([]);
  const [venues, setVenues] = useState<Venue[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const dropdownRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (
        dropdownRef.current &&
        !dropdownRef.current.contains(event.target as Node)
      ) {
        onClose();
      }
    };

    if (isOpen) {
      document.addEventListener("mousedown", handleClickOutside);
    }

    return () => {
      document.removeEventListener("mousedown", handleClickOutside);
    };
  }, [isOpen, onClose]);

  useEffect(() => {
    if (!searchQuery.trim()) {
      setTreatments([]);
      setVenues([]);
      return;
    }

    const searchDelay = setTimeout(async () => {
      setIsLoading(true);
      try {
        // Get service types and filter by search query
        const serviceTypes = await getServiceTypesClient();
        const matchingServices = serviceTypes
          .filter((service) =>
            service.typeName.toLowerCase().includes(searchQuery.toLowerCase())
          )
          .slice(0, 5)
          .map((service) => ({
            name: service.typeName,
            type: "treatment" as const,
          }));
        setTreatments(matchingServices);

        // Get venues from home API (always fetch, even if location is not set)
        if (location) {
          const homeRequest = {
            targetLat: location.lat,
            targetLng: location.lng,
            rangeInKm: 500,
            serviceType: null,
          };

          const response = await getHomeVenuesClient(homeRequest);
          const transformedVenues = response.slons
            .filter((salon) =>
              salon.salonName.toLowerCase().includes(searchQuery.toLowerCase())
            )
            .slice(0, 5)
            .map(transformSalonToVenue);
          setVenues(transformedVenues);
        } else {
          setVenues([]);
        }
      } catch (error) {
        console.error("Failed to fetch search suggestions:", error);
        setTreatments([]);
        setVenues([]);
      } finally {
        setIsLoading(false);
      }
    }, 300);

    return () => clearTimeout(searchDelay);
  }, [searchQuery, location]);

  const getImageUrl = (imagePath: string) => {
    if (imagePath.startsWith("http://") || imagePath.startsWith("https://")) {
      return imagePath;
    }
    return imagePath.startsWith("/")
      ? `${API_BASE_URL}${imagePath}`
      : `${API_BASE_URL}/${imagePath}`;
  };

  if (!isOpen || !searchQuery.trim()) return null;

  return (
    <div
      ref={dropdownRef}
      className="absolute top-full left-0 mt-2 bg-white rounded-xl shadow-lg border border-gray-200 z-50 max-h-96 overflow-y-auto"
      style={{ width: "312px" }}
    >
      {/* Treatments Section */}
      {treatments.length > 0 && (
        <div className="p-4">
          <h3 className="heading-6 text-gray-900 mb-2">Treatments</h3>
          <div className="space-y-1">
            {treatments.map((treatment, index) => (
              <button
                key={index}
                onClick={onClose}
                className="w-full px-4 py-2 text-left hover:bg-gray-50 rounded-lg flex items-center gap-3"
              >
                <div className="w-8 h-8 rounded-lg bg-purple-50 border border-purple-200 flex items-center justify-center">
                  <Search className="w-4 h-4 text-purple-600" />
                </div>
                <span className="body-text">{treatment.name}</span>
              </button>
            ))}
          </div>
        </div>
      )}

      {/* Venues Section */}
      {venues.length > 0 && (
        <div className={`p-4 ${treatments.length > 0 ? "border-t border-gray-200" : ""}`}>
          <h3 className="heading-6 text-gray-900 mb-2">Venues</h3>
          <div className="space-y-2">
            {venues.map((venue) => (
              <Link
                key={venue.id}
                href={routes.saloons}
                className="block px-4 py-2 hover:bg-gray-50 rounded-lg"
                onClick={onClose}
              >
                <div className="flex items-center gap-3">
                  <div className="w-12 h-12 rounded-lg overflow-hidden flex-shrink-0">
                    {venue.image && !venue.image.includes("placeholder") ? (
                      <Image
                        src={venue.image}
                        alt={venue.name}
                        width={48}
                        height={48}
                        className="w-full h-full object-cover"
                      />
                    ) : (
                      <div className="w-full h-full bg-gray-100 flex items-center justify-center">
                        <MapPin className="w-5 h-5 text-gray-400" />
                      </div>
                    )}
                  </div>
                  <div className="flex-1 min-w-0">
                    <h4 className="body-text font-medium text-gray-900 line-clamp-1">
                      {venue.name}
                    </h4>
                    <p className="body-text-sm text-gray-500 line-clamp-1">
                      {venue.category}
                    </p>
                    <p className="body-text-sm text-gray-400 line-clamp-1">
                      {venue.address || venue.location}
                    </p>
                  </div>
                </div>
              </Link>
            ))}
          </div>
        </div>
      )}

      {isLoading && (
        <div className="p-8 text-center text-gray-500">Searching...</div>
      )}

      {!isLoading && treatments.length === 0 && venues.length === 0 && (
        <div className="p-8 text-center text-gray-500">No results found</div>
      )}
    </div>
  );
}

