/**
 * Search Salons API - Server Side Only
 * Searches salons using the customer/search endpoint
 */

import "server-only";
import { z } from "zod";
import { HomeSalonSchema, type HomeSalon } from "@/lib/schemas/home";

const SearchSalonsResponseSchema = z.object({
  status: z.string(),
  message: z.string(),
  data: z.object({
    slons: z.array(HomeSalonSchema),
    appointments: z.array(z.unknown()).optional(),
    currencyUnit: z.string(),
    distanceUnit: z.string(),
    highestPriceAcrossAllSalons: z.string(),
    lowestPriceAcrossAllSalons: z.string(),
  }),
  error: z.string(),
});

export interface SearchSalonsRequest {
  targetLat: number;
  targetLng: number;
  rangeInKm?: number;
  day?: string;
  time?: string;
  price?: number;
}

export interface SearchSalonsResponse {
  slons: HomeSalon[];
  currencyUnit: string;
  distanceUnit: string;
  highestPriceAcrossAllSalons: string;
  lowestPriceAcrossAllSalons: string;
}

/**
 * Server-side: Search salons
 */
export async function searchSalonsServer(
  request: SearchSalonsRequest
): Promise<SearchSalonsResponse> {
  const { serverFetcher } = await import("@/lib/api/fetcher-server");

  const data = await serverFetcher<unknown>("/customer/search", {
    requireAuth: false,
    method: "POST",
    body: JSON.stringify(request),
  });

  const parsed = SearchSalonsResponseSchema.parse(data);
  return parsed.data;
}






