"use client";

import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { StarRating } from "@/components/data-display/StarRating";
import { Employee } from "@/lib/schemas/salonDetail";
import { getImageUrl } from "../utils/imageUtils";

interface SalonTeamProps {
  employees?: Employee[];
}

export function SalonTeam({ employees = [] }: SalonTeamProps) {
  if (employees.length === 0) {
    return null;
  }

  const getInitials = (firstName?: string, lastName?: string): string => {
    const first = firstName?.charAt(0).toUpperCase() || "";
    const last = lastName?.charAt(0).toUpperCase() || "";
    return first + last || "?";
  };

  return (
    <div className="space-y-4">
      <h2 className="text-lg md:text-3xl font-bold">Team</h2>
      <div className="flex gap-16 flex-wrap pb-2 scrollbar-hide">
        {employees.map((employee) => {
          const user = employee.user;
          const firstName = user?.firstName || "";
          const lastName = user?.lastName || "";
          const fullName = `${firstName} ${lastName}`.trim() || "Team Member";
          const position = employee.position || "";
          const rating = employee.employeeAverageRating
            ? parseFloat(employee.employeeAverageRating)
            : null;
          const hasImage = user?.image && user.image.trim() !== "";

          return (
            <div
              key={employee.id}
              className="flex flex-wrap flex-col items-center gap-2 min-w-[80px] md:min-w-[100px]"
            >
              <div className="relative mb-3">
                <Avatar className="w-20 h-20 md:w-32 md:h-32">
                  <AvatarImage
                    src={hasImage ? getImageUrl(user.image) : undefined}
                    alt={fullName}
                  />
                  <AvatarFallback className="bg-purple-100 text-purple-700 text-base md:text-xl font-semibold">
                    {getInitials(firstName, lastName)}
                  </AvatarFallback>
                </Avatar>
                {rating !== null && rating > 0 && (
                  <div className="absolute bottom-0 left-1/2 -translate-x-1/2 translate-y-1/2 z-10">
                    <div className="bg-white border border-purple-200 rounded-full px-2 md:px-2.5 py-0.5 md:py-1 flex items-center gap-1 md:gap-1.5 whitespace-nowrap">
                      <span className="text-[10px] md:text-xs font-semibold text-gray-900 leading-none">
                        {rating.toFixed(1)}
                      </span>
                      <span className="text-yellow-400 text-[10px] md:text-xs leading-none">
                        ★
                      </span>
                    </div>
                  </div>
                )}
              </div>
              <div className="text-center">
                <p className="font-bold text-xs md:text-sm text-gray-900">{fullName}</p>
                {position && (
                  <p className="text-xs text-gray-500 mt-0.5">{position}</p>
                )}
              </div>
            </div>
          );
        })}
      </div>
    </div>
  );
}
