"use client";

import { useState, useRef } from "react";
import { useRouter } from "next/navigation";
import { Button } from "@/components/ui/button";
import { Dialog } from "@/components/ui/dialog";
import { Category, Service } from "@/lib/schemas/salonDetail";
import { ChevronLeft, ChevronRight } from "lucide-react";
import { routes } from "@/config/routes";

interface SalonServicesProps {
  categories?: Category[];
  currencyUnit?: string;
  salonId?: string | number;
}

export function SalonServices({
  categories = [],
  currencyUnit = "£",
  salonId,
}: SalonServicesProps) {
  const router = useRouter();
  // Filter out categories that have no services
  const categoriesWithServices = categories.filter(
    (category) => category.services && category.services.length > 0
  );

  const [selectedCategory, setSelectedCategory] = useState<string | null>(
    categoriesWithServices[0]?.categoryName || null
  );
  const [selectedService, setSelectedService] = useState<Service | null>(null);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const scrollContainerRef = useRef<HTMLDivElement>(null);

  const scrollLeft = () => {
    if (scrollContainerRef.current) {
      scrollContainerRef.current.scrollBy({ left: -200, behavior: "smooth" });
    }
  };

  const scrollRight = () => {
    if (scrollContainerRef.current) {
      scrollContainerRef.current.scrollBy({ left: 200, behavior: "smooth" });
    }
  };

  // Get all services from selected category
  const selectedCategoryData = categoriesWithServices.find(
    (cat) => cat.categoryName === selectedCategory
  );
  const services = selectedCategoryData?.services || [];

  // Format duration in minutes to readable format
  const formatDuration = (minutes?: number | null): string => {
    if (!minutes) return "";
    if (minutes < 60) return `${minutes} mins`;
    const hours = Math.floor(minutes / 60);
    const mins = minutes % 60;
    if (mins === 0) return `${hours} hr${hours > 1 ? "s" : ""}`;
    return `${hours} hr ${mins} mins`;
  };

  // Calculate price range for a service
  const getPriceRange = (service: Service): string => {
    const price = parseFloat(service.price || "0");
    if (price === 0) return "";
    const discount = service.discount || 0;
    const discountedPrice = price * (1 - discount / 100);
    const finalPrice = discount > 0 ? discountedPrice : price;
    return `from ${currencyUnit}${finalPrice.toFixed(2)}`;
  };

  // Get exact price (without "from")
  const getExactPrice = (service: Service): string => {
    const price = parseFloat(service.price || "0");
    if (price === 0) return "";
    const discount = service.discount || 0;
    const discountedPrice = price * (1 - discount / 100);
    const finalPrice = discount > 0 ? discountedPrice : price;
    return `${currencyUnit}${finalPrice.toFixed(2)}`;
  };

  const handleServiceClick = (service: Service) => {
    setSelectedService(service);
    setIsModalOpen(true);
  };

  const handleAddToBooking = () => {
    // TODO: Implement add to booking logic
    console.log("Add to booking:", selectedService);
    setIsModalOpen(false);
  };

  return (
    <div className="space-y-4">
      <h2 className="text-lg md:text-3xl font-bold">Services</h2>

      {/* Category Tabs */}
      {categoriesWithServices.length > 0 && (
        <div className="flex items-center gap-2">
          <div
            ref={scrollContainerRef}
            className="flex gap-2 overflow-x-auto pb-2 scrollbar-hide flex-1"
          >
            {categoriesWithServices.map((category) => (
              <button
                key={category.id}
                onClick={() =>
                  setSelectedCategory(category.categoryName || null)
                }
                className={`px-4 py-2 rounded-full text-sm font-medium whitespace-nowrap transition-colors ${
                  selectedCategory === category.categoryName
                    ? "bg-gray-900 text-white"
                    : "bg-gray-100 text-gray-700 hover:bg-gray-200"
                }`}
              >
                {category.categoryName || "Category"}
              </button>
            ))}
          </div>
          <div className="flex items-center gap-1 flex-shrink-0">
            <button
              onClick={scrollLeft}
              className="p-1.5 rounded-full hover:bg-gray-100 transition-colors"
              aria-label="Scroll left"
            >
              <ChevronLeft className="w-5 h-5 text-gray-600" />
            </button>
            <button
              onClick={scrollRight}
              className="p-1.5 rounded-full hover:bg-gray-100 transition-colors"
              aria-label="Scroll right"
            >
              <ChevronRight className="w-5 h-5 text-gray-600" />
            </button>
          </div>
        </div>
      )}

      {/* Services List */}
      <div className="space-y-3">
        {services.length === 0 ? (
          <p className="text-gray-500 text-sm">
            No services available in this category.
          </p>
        ) : (
          services.map((service) => {
            const duration = formatDuration(service.duration);
            const priceRange = getPriceRange(service);

            return (
              <div
                key={service.id}
                className="flex items-center justify-between gap-4 p-4 border border-gray-200 rounded-lg hover:border-gray-300 transition-colors cursor-pointer"
                onClick={() => handleServiceClick(service)}
              >
                <div className="flex-1 min-w-0">
                  {/* Service Name - Top, largest and boldest */}
                  <h3 className="font-semibold md:font-semibold text-gray-900 text-base md:text-base">
                    {service.serviceName || "Service"}
                  </h3>

                  {/* Duration - Second line, lighter gray */}
                  {duration && (
                    <p className="text-xs md:text-sm text-gray-500 mb-2 md:mb-3">{duration}</p>
                  )}

                  {/* Price - Third line, dark gray */}
                  {priceRange && (
                    <p className="text-xs md:text-sm font-medium text-gray-900">
                      {priceRange}
                    </p>
                  )}
                </div>
                <Button
                  size="sm"
                  variant="primary"
                  onClick={(e) => {
                    e.stopPropagation();
                    if (salonId) {
                      router.push(routes.salonServices(salonId));
                    } else {
                      handleServiceClick(service);
                    }
                  }}
                >
                  Book
                </Button>
              </div>
            );
          })
        )}
      </div>

      {/* Service Modal */}
      <Dialog open={isModalOpen} onOpenChange={setIsModalOpen}>
        {selectedService && (
          <div className="p-6">
            {/* Service Name - Large, bold, top */}
            <h2 className="heading-3 font-bold text-gray-900 mb-4">
              {selectedService.serviceName || "Service"}
            </h2>

            {/* Description - Below name */}
            {selectedService.description && (
              <p className="body-text text-gray-700 mb-6 leading-relaxed">
                {selectedService.description}
              </p>
            )}

            {/* Price - Below description */}
            {selectedService.price && (
              <p className="text-lg font-semibold text-gray-900 mb-2">
                {getExactPrice(selectedService)}
              </p>
            )}

            {/* Duration - Below price */}
            {selectedService.duration && (
              <p className="text-sm text-gray-600 mb-6">
                {formatDuration(selectedService.duration)}
              </p>
            )}

            {/* Add to booking button */}
            <Button
              variant="primary"
              size="lg"
              className="w-full"
              onClick={handleAddToBooking}
            >
              Add to booking
            </Button>
          </div>
        )}
      </Dialog>

      {services.length > 0 && (
        <button className="text-xs md:text-sm text-blue-600 hover:underline">
          See all
        </button>
      )}
    </div>
  );
}
