"use client";

import { useState } from "react";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Booking } from "@/lib/schemas/salonDetail";
import { getImageUrl } from "../utils/imageUtils";

interface SalonReviewsProps {
  bookings?: Booking[];
  averageRating?: number | null;
  reviewCount?: number;
}

// Component to render star rating with filled/empty stars
function StarDisplay({
  rating,
  size = "md",
}: {
  rating: number;
  size?: "sm" | "md" | "lg" | "xl";
}) {
  const fullStars = Math.floor(rating);
  const hasHalfStar = rating % 1 >= 0.5;
  const emptyStars = 5 - fullStars - (hasHalfStar ? 1 : 0);

  const sizeClasses = {
    sm: "text-base md:text-lg", // 16px mobile, 18px desktop for individual reviews
    md: "text-lg md:text-xl", // 18px mobile, 20px desktop
    lg: "text-xl md:text-2xl", // 20px mobile, 24px desktop
    xl: "text-2xl md:text-4xl", // 24px mobile, 36px desktop for overall rating
  };

  return (
    <div className={`flex items-center gap-1 ${sizeClasses[size]}`}>
      {[...Array(fullStars)].map((_, i) => (
        <span key={i} className="text-yellow-400 leading-none">
          ★
        </span>
      ))}
      {hasHalfStar && <span className="text-yellow-400 leading-none">★</span>}
      {[...Array(emptyStars)].map((_, i) => (
        <span key={i} className="text-yellow-400 opacity-30 leading-none">
          ★
        </span>
      ))}
    </div>
  );
}

export function SalonReviews({
  bookings = [],
  averageRating,
  reviewCount = 0,
}: SalonReviewsProps) {
  const [expandedReviews, setExpandedReviews] = useState<Set<number>>(
    new Set()
  );
  const [showAll, setShowAll] = useState(false);

  // Include all bookings with ratings, even if no feedback
  const reviews = bookings.filter(
    (booking) => booking.averageRatingPerBooking || booking.feedback
  );
  const displayedReviews = showAll ? reviews : reviews.slice(0, 6);

  const toggleExpanded = (id?: number) => {
    if (id === undefined) return;
    setExpandedReviews((prev) => {
      const next = new Set(prev);
      if (next.has(id)) {
        next.delete(id);
      } else {
        next.add(id);
      }
      return next;
    });
  };

  const getInitials = (firstName?: string, lastName?: string): string => {
    const first = firstName?.charAt(0).toUpperCase() || "";
    const last = lastName?.charAt(0).toUpperCase() || "";
    return first + last || "?";
  };

  const formatReviewDate = (dateString?: string): string => {
    if (!dateString) return "";
    try {
      const date = new Date(dateString);
      const day = date.toLocaleDateString("en-US", { weekday: "short" });
      const month = date.toLocaleDateString("en-US", { month: "short" });
      const dayNum = date.getDate();
      const year = date.getFullYear();
      const time = date
        .toLocaleTimeString("en-US", {
          hour: "numeric",
          minute: "2-digit",
          hour12: true,
        })
        .toLowerCase();
      return `${day}, ${dayNum} ${month} ${year} at ${time}`;
    } catch {
      return "";
    }
  };

  return (
    <div className="space-y-6">
      {/* Header with overall rating */}
      <div className="space-y-3">
        <h2 className="text-lg md:text-3xl font-bold">Reviews</h2>
        {averageRating !== null && averageRating !== undefined && (
          <div className="flex items-center gap-3 md:gap-4">
            <StarDisplay rating={averageRating} size="xl" />
            <span className="text-lg md:text-2xl font-bold text-gray-900">
              {averageRating.toFixed(1)} ({reviewCount})
            </span>
          </div>
        )}
      </div>

      {displayedReviews.length === 0 ? (
        <p className="text-gray-500 text-sm">No reviews yet.</p>
      ) : (
        <div className="grid grid-cols-1 md:grid-cols-2 gap-x-16 gap-y-12">
          {displayedReviews.map((review) => {
            const user = review.user;
            const firstName = user?.firstName || "";
            const lastName = user?.lastName || "";
            const fullName = `${firstName} ${lastName}`.trim() || "Anonymous";
            const feedback = review.feedback || "";
            const rating = review.averageRatingPerBooking
              ? parseFloat(review.averageRatingPerBooking)
              : null;
            const isExpanded = expandedReviews.has(review.id || 0);
            const shouldTruncate = feedback.length > 150;

            return (
              <div key={review.id} className="space-y-1.5">
                {/* First row: Avatar and Name/Date side by side */}
                <div className="flex gap-2 md:gap-3 items-center">
                  <Avatar className="w-12 h-12 md:w-16 md:h-16 flex-shrink-0">
                    <AvatarImage
                      src={user?.image ? getImageUrl(user.image) : undefined}
                      alt={fullName}
                    />
                    <AvatarFallback className="bg-purple-100 text-purple-700 text-sm md:text-lg font-semibold">
                      {getInitials(firstName, lastName)}
                    </AvatarFallback>
                  </Avatar>

                  {/* Name and date - vertically centered with circle */}
                  <div className="flex-1 min-w-0">
                    <p className="text-sm md:text-base font-semibold text-gray-900">
                      {fullName}
                    </p>
                    <p className="text-xs md:text-sm text-gray-500">
                      {formatReviewDate(review.createdAt ?? undefined)}
                    </p>
                  </div>
                </div>

                {/* Stars - below circle, left aligned with circle */}
                {/* Always render this div to maintain spacing, even if no rating */}
                <div className="ml-0" style={{ minHeight: "1.5rem" }}>
                  {rating !== null && rating > 0 && (
                    <StarDisplay rating={rating} size="sm" />
                  )}
                </div>

                {/* Feedback - below circle, left aligned with circle */}
                {feedback && (
                  <p className="text-sm md:text-base text-gray-700 leading-relaxed">
                    {isExpanded || !shouldTruncate
                      ? feedback
                      : `${feedback.substring(0, 150)}...`}
                    {shouldTruncate && (
                      <button
                        onClick={() => toggleExpanded(review.id)}
                        className="text-blue-600 hover:underline ml-1 font-medium"
                      >
                        {isExpanded ? "Read less" : "Read more"}
                      </button>
                    )}
                  </p>
                )}
              </div>
            );
          })}
        </div>
      )}

      {reviews.length > 6 && (
        <div className="flex justify-start pt-4">
          <button
            onClick={() => setShowAll(!showAll)}
            className="px-4 md:px-6 py-2 md:py-2.5 text-xs md:text-sm font-semibold text-blue-600 border-2 border-blue-600 rounded-lg hover:bg-blue-50 hover:border-blue-700 transition-colors duration-200"
          >
            {showAll ? "Show less" : "See all reviews"}
          </button>
        </div>
      )}
    </div>
  );
}
