"use client";

import { useState, useEffect } from "react";
import { Time } from "@/lib/schemas/salonDetail";

interface SalonOpeningTimesProps {
  times?: Time[];
}

export function SalonOpeningTimes({ times = [] }: SalonOpeningTimesProps) {
  const [today, setToday] = useState<string>("");

  useEffect(() => {
    // Only set current day on client side to avoid hydration mismatch
    setToday(new Date().toLocaleDateString("en-US", { weekday: "long" }));
  }, []);

  const formatTime = (timeString?: string | null): string => {
    if (!timeString) return "Closed";
    try {
      const date = new Date(`2000-01-01T${timeString}`);
      return date.toLocaleTimeString("en-US", {
        hour: "numeric",
        minute: "2-digit",
        hour12: true,
      });
    } catch {
      return timeString;
    }
  };

  const dayOrder = [
    "Monday",
    "Tuesday",
    "Wednesday",
    "Thursday",
    "Friday",
    "Saturday",
    "Sunday",
  ];

  const sortedTimes = dayOrder.map((day) => {
    const time = times.find((t) => t.day === day);
    return {
      day,
      ...time,
    };
  });

  return (
    <div className="space-y-4">
      <h2 className="text-lg md:text-3xl font-bold">Opening times</h2>
      <div className="space-y-2">
        {sortedTimes.map((time) => {
          const isOpen = time.status && time.openingTime && time.closingTime;
          const hours = isOpen
            ? `${formatTime(time.openingTime)} - ${formatTime(
                time.closingTime
              )}`
            : "Closed";
          const isToday = time.day === today;
          const isClosed = !isOpen;

          return (
            <div
              key={time.day}
              className="flex items-center gap-3 py-2 border-b border-gray-100 last:border-0"
            >
              {/* Green dot for open days, gray dot for closed days */}
              {isOpen ? (
                <div className="w-2 h-2 bg-green-500 rounded-full flex-shrink-0" />
              ) : (
                <div className="w-2 h-2 bg-gray-400 rounded-full flex-shrink-0" />
              )}
              <span
                className={`text-xs md:text-sm flex-1 ${
                  isToday
                    ? "font-bold text-gray-900"
                    : isClosed
                    ? "text-gray-400 font-medium"
                    : "font-medium text-gray-900"
                }`}
              >
                {time.day}
              </span>
              <span
                className={`text-xs md:text-sm ${
                  isToday
                    ? "font-bold text-gray-900"
                    : isClosed
                    ? "text-gray-400"
                    : "text-gray-900 font-medium"
                }`}
              >
                {hours}
              </span>
            </div>
          );
        })}
      </div>
    </div>
  );
}
