"use client";

import { useEffect, useRef } from "react";
import { MapPin } from "lucide-react";
import { AddressDB } from "@/lib/schemas/salonDetail";
import { googleApiKey } from "@/config/env";

interface SalonMapProps {
  address?: AddressDB;
  salonName?: string;
}

export function SalonMap({ address, salonName }: SalonMapProps) {
  const mapRef = useRef<HTMLDivElement>(null);
  const mapInstanceRef = useRef<google.maps.Map | null>(null);

  console.log("Current Maps Key:", googleApiKey); // Debugging line

  useEffect(() => {
    if (!mapRef.current || !address?.lat || !address?.lng) return;

    const lat = parseFloat(address.lat);
    const lng = parseFloat(address.lng);

    if (isNaN(lat) || isNaN(lng)) return;

    function initializeMap() {
      if (!mapRef.current || !window.google) return;

      const map = new window.google.maps.Map(mapRef.current, {
        center: { lat, lng },
        zoom: 15,
        mapTypeControl: false,
        fullscreenControl: false,
        streetViewControl: false,
      });

      new window.google.maps.Marker({
        position: { lat, lng },
        map,
        title: salonName || "Salon location",
      });

      mapInstanceRef.current = map;
    }

    // Load Google Maps script if not already loaded
    if (!window.google) {
      // Check if script is already being loaded
      const existingScript = document.querySelector(
        `script[src*="maps.googleapis.com"]`
      );

      if (existingScript) {
        existingScript.addEventListener("load", initializeMap);
      } else {
        const script = document.createElement("script");
        script.src = `https://maps.googleapis.com/maps/api/js?key=${googleApiKey}&libraries=places`;
        script.async = true;
        script.defer = true;
        script.onload = initializeMap;
        document.head.appendChild(script);
      }
    } else {
      initializeMap();
    }

    return () => {
      if (mapInstanceRef.current) {
        mapInstanceRef.current = null;
      }
    };
  }, [address?.lat, address?.lng, salonName]);

  if (!address?.lat || !address?.lng) {
    return (
      <div className="w-full h-64 bg-gray-100 rounded-lg flex items-center justify-center">
        <p className="text-gray-500 text-sm">Map not available</p>
      </div>
    );
  }

  const fullAddress = [
    address.building,
    address.streetAddress,
    address.city,
    address.district,
    address.province,
    address.country,
  ]
    .filter(Boolean)
    .join(", ");

  return (
    <div className="space-y-4">
      <h2 className="text-lg md:text-3xl font-bold">Location</h2>
      <div className="space-y-2">
        <div ref={mapRef} className="w-full h-64 rounded-lg overflow-hidden" />
        {fullAddress && (
          <div className="flex items-start gap-2">
            <MapPin className="w-4 h-4 md:w-5 md:h-5 text-gray-500 mt-0.5 flex-shrink-0" />
            <div className="flex-1">
              <p className="text-xs md:text-sm text-gray-700">{fullAddress}</p>
              <button className="text-xs md:text-sm text-blue-600 hover:underline mt-1">
                Get directions
              </button>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}

