"use client";

import { StarRating } from "@/components/data-display/StarRating";
import { Button } from "@/components/ui/button";
import { Heart, Share2, MapPin, Clock, Check } from "lucide-react";
import { SalonDetail } from "@/lib/schemas/salonDetail";
import { getImageUrl } from "../utils/imageUtils";
import { useState, useEffect } from "react";
import { useLikeDislikeSalon } from "../hooks/useLikeDislikeSalon";

interface SalonDetailHeaderProps {
  salon: SalonDetail;
  currencyUnit?: string;
  isLiked?: boolean;
  onLikeToggle?: () => void;
}

export function SalonDetailHeader({
  salon,
  currencyUnit = "£",
  isLiked = false,
  onLikeToggle,
}: SalonDetailHeaderProps) {
  const [liked, setLiked] = useState(isLiked);
  const [openUntil, setOpenUntil] = useState<string | null>(null);
  const [isOpen, setIsOpen] = useState(false);
  const [shareCopied, setShareCopied] = useState(false);

  // Debug log in development
  useEffect(() => {
    if (process.env.NODE_ENV === "development") {
      console.log("[SalonDetailHeader] isLiked prop:", isLiked, "liked state:", liked);
    }
  }, [isLiked, liked]);

  // Sync liked state with prop changes (e.g., when favorites API refetches)
  useEffect(() => {
    setLiked(isLiked);
  }, [isLiked]);

  // Like/Dislike mutation
  const likeDislikeMutation = useLikeDislikeSalon();

  useEffect(() => {
    // Only calculate opening hours on client side to avoid hydration mismatch
    const today = new Date().toLocaleDateString("en-US", { weekday: "long" });
    const todayHours = salon.times?.find((t) => t.day === today);
    const open =
      todayHours?.status && todayHours.openingTime && todayHours.closingTime;
    setIsOpen(!!open);

    if (open && todayHours?.closingTime) {
      const time = new Date(
        `2000-01-01T${todayHours.closingTime}`
      ).toLocaleTimeString("en-US", {
        hour: "numeric",
        minute: "2-digit",
        hour12: true,
      });
      setOpenUntil(time);
    } else {
      setOpenUntil(null);
    }
  }, [salon.times]);

  const handleLike = async () => {
    if (!salon.id) {
      console.error("Salon ID is missing");
      return;
    }

    // Store previous state for error revert
    const previousLiked = liked;
    
    // Optimistic update
    const newLikedState = !liked;
    setLiked(newLikedState);

    try {
      // Call the API to toggle like/dislike
      const response = await likeDislikeMutation.mutateAsync({
        salonId: salon.id,
      });
      
      // Update local state based on API response
      if (response.data?.like !== undefined) {
        setLiked(response.data.like);
      } else {
        // If response doesn't have like status, keep optimistic update
        setLiked(newLikedState);
      }
      
      onLikeToggle?.();
    } catch (error) {
      // Revert optimistic update on error
      setLiked(previousLiked);
      console.error("Failed to toggle favorite:", error);
    }
  };

  const handleShare = async () => {
    const url = window.location.href;
    const title = salon.salonName || "Salon";
    const text = `Check out ${title} on Fresha`;

    try {
      // Try Web Share API first (works on mobile and modern browsers)
      if (
        typeof navigator !== "undefined" &&
        "share" in navigator &&
        typeof navigator.share === "function"
      ) {
        await navigator.share({
          title,
          text,
          url,
        });
        return;
      }

      // Fallback: Copy to clipboard
      await navigator.clipboard.writeText(url);
      setShareCopied(true);
      setTimeout(() => setShareCopied(false), 2000);
    } catch (error) {
      // User cancelled share or error occurred
      // If Web Share API failed, try clipboard fallback
      if (
        typeof navigator !== "undefined" &&
        "share" in navigator &&
        typeof navigator.share === "function"
      ) {
        try {
          await navigator.clipboard.writeText(url);
          setShareCopied(true);
          setTimeout(() => setShareCopied(false), 2000);
        } catch (clipboardError) {
          console.error("Failed to copy to clipboard:", clipboardError);
        }
      }
    }
  };

  const address = salon.addressDB;
  const fullAddress = [
    address?.building,
    address?.streetAddress,
    address?.city,
    address?.district,
    address?.province,
    address?.country,
  ]
    .filter(Boolean)
    .join(", ");

  const rating = salon.salonAverageRating ?? 0;
  const reviewCount = salon.ratingCount ?? 0;

  // Get location for display - prefer district, then city
  const location = address?.district
    ? `${address.district}${address.city ? `, ${address.city}` : ""}`
    : address?.city || "";

  // Star display component
  const StarDisplay = ({ rating }: { rating: number }) => {
    const fullStars = Math.floor(rating);
    const hasHalfStar = rating % 1 >= 0.5;
    const emptyStars = 5 - fullStars - (hasHalfStar ? 1 : 0);

    return (
      <div className="flex items-center gap-0.5">
        {[...Array(fullStars)].map((_, i) => (
          <span key={i} className="text-yellow-400 text-base md:text-lg leading-none">
            ★
          </span>
        ))}
        {hasHalfStar && (
          <span className="text-yellow-400 text-base md:text-lg leading-none">★</span>
        )}
        {[...Array(emptyStars)].map((_, i) => (
          <span
            key={i}
            className="text-yellow-400 text-base md:text-lg opacity-30 leading-none"
          >
            ★
          </span>
        ))}
      </div>
    );
  };

  return (
    <div className="space-y-2 md:space-y-3">
      {/* Title with Action Buttons */}
      <div className="flex items-center justify-between gap-4">
        <h1 className="text-xl md:text-3xl font-bold md:font-bold flex-1 leading-tight">{salon.salonName || "Salon"}</h1>

        {/* Action buttons inline with salon name */}
        <div className="flex items-center gap-2 flex-shrink-0">
          <button
            onClick={handleShare}
            className="w-10 h-10 rounded-full border border-gray-300 flex items-center justify-center hover:bg-gray-50 transition-colors relative"
            aria-label="Share"
          >
            {shareCopied ? (
              <Check className="w-5 h-5 text-green-600" />
            ) : (
              <Share2 className="w-5 h-5 text-gray-900" />
            )}
          </button>
          <button
            onClick={handleLike}
            disabled={likeDislikeMutation.isPending || !salon.id}
            className="w-10 h-10 rounded-full border border-gray-300 flex items-center justify-center hover:bg-gray-50 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
            aria-label={liked ? "Remove from favorites" : "Add to favorites"}
          >
            <Heart
              className={`w-5 h-5 ${
                liked ? "fill-red-500 text-red-500" : "text-gray-900"
              }`}
            />
          </button>
        </div>
      </div>

      {/* Stacked layout: Rating, Location, Open status */}
      <div className="space-y-1.5 md:space-y-2">
        {/* Rating row */}
        {rating > 0 && (
          <div className="flex items-center gap-1.5 md:gap-2">
            <span className="text-base md:text-lg font-semibold text-gray-900">
              {rating.toFixed(1)}
            </span>
            <StarDisplay rating={rating} />
            <span className="text-sm md:text-base text-purple-600 font-medium">
              ({reviewCount})
            </span>
          </div>
        )}
        
        {/* Location row */}
        {location && (
          <div className="flex items-center gap-1.5 md:gap-2 text-xs md:text-sm text-gray-600">
            <span>{location}</span>
            <button className="text-purple-600 hover:underline font-medium">
              Get directions
            </button>
          </div>
        )}
        
        {/* Open status row */}
        {isOpen && openUntil && (
          <span className="text-xs md:text-sm text-green-600 font-medium block">
            Open until {openUntil}
          </span>
        )}
      </div>
    </div>
  );
}
