"use client";

import { useState, useEffect, useRef } from "react";
import { Button } from "@/components/ui/button";
import { StarRating } from "@/components/data-display/StarRating";
import { SalonDetail } from "@/lib/schemas/salonDetail";
import { Clock, MapPin, ChevronDown, ChevronUp } from "lucide-react";

interface SalonBookingCardProps {
  salon: SalonDetail;
  currencyUnit?: string;
  onBookClick?: () => void;
}

export function SalonBookingCard({
  salon,
  currencyUnit = "£",
  onBookClick,
}: SalonBookingCardProps) {
  const rating = salon.salonAverageRating ?? 0;
  const reviewCount = salon.ratingCount ?? 0;
  const [openUntil, setOpenUntil] = useState<string | null>(null);
  const [isOpen, setIsOpen] = useState(false);
  const [showAllHours, setShowAllHours] = useState(false);
  const [today, setToday] = useState<string>("");
  const [showHeaderInfo, setShowHeaderInfo] = useState(false);
  const headerObserverRef = useRef<IntersectionObserver | null>(null);

  useEffect(() => {
    // Only calculate opening hours on client side to avoid hydration mismatch
    const currentDay = new Date().toLocaleDateString("en-US", {
      weekday: "long",
    });
    setToday(currentDay);
    const todayHours = salon.times?.find((t) => t.day === currentDay);
    const open =
      todayHours?.status && todayHours.openingTime && todayHours.closingTime;
    setIsOpen(!!open);

    if (open && todayHours?.closingTime) {
      const time = new Date(
        `2000-01-01T${todayHours.closingTime}`
      ).toLocaleTimeString("en-US", {
        hour: "numeric",
        minute: "2-digit",
        hour12: true,
      });
      setOpenUntil(time);
    } else {
      setOpenUntil(null);
    }
  }, [salon.times]);

  // Detect when header goes out of view
  useEffect(() => {
    const checkHeaderVisibility = () => {
      const headerElement = document.getElementById("salon-detail-header");
      if (!headerElement) {
        setShowHeaderInfo(false);
        return;
      }

      const rect = headerElement.getBoundingClientRect();
      // Show info in booking card when header has scrolled above the viewport
      const isHeaderOutOfView = rect.bottom < 0;
      setShowHeaderInfo(isHeaderOutOfView);
    };

    // Check initial state
    checkHeaderVisibility();

    // Use Intersection Observer for efficient detection
    const headerElement = document.getElementById("salon-detail-header");
    if (headerElement) {
      headerObserverRef.current = new IntersectionObserver(
        (entries) => {
          // If header is not intersecting (out of view), show info in booking card
          setShowHeaderInfo(!entries[0].isIntersecting);
        },
        {
          threshold: 0,
          rootMargin: "0px",
        }
      );

      headerObserverRef.current.observe(headerElement);
    }

    // Also listen to scroll for immediate updates
    window.addEventListener("scroll", checkHeaderVisibility, { passive: true });

    return () => {
      if (headerObserverRef.current) {
        headerObserverRef.current.disconnect();
      }
      window.removeEventListener("scroll", checkHeaderVisibility);
    };
  }, []);

  const formatTime = (timeString?: string | null): string => {
    if (!timeString) return "Closed";
    try {
      const date = new Date(`2000-01-01T${timeString}`);
      return date.toLocaleTimeString("en-US", {
        hour: "numeric",
        minute: "2-digit",
        hour12: true,
      });
    } catch {
      return timeString;
    }
  };

  const dayOrder = [
    "Monday",
    "Tuesday",
    "Wednesday",
    "Thursday",
    "Friday",
    "Saturday",
    "Sunday",
  ];

  const sortedTimes = dayOrder.map((day) => {
    const time = salon.times?.find((t) => t.day === day);
    return {
      day,
      ...time,
    };
  });

  const address = salon.addressDB;
  const fullAddress = [
    address?.building,
    address?.streetAddress,
    address?.city,
    address?.district,
    address?.province,
    address?.country,
  ]
    .filter(Boolean)
    .join(", ");

  // Star display component matching the header
  const StarDisplay = ({ rating }: { rating: number }) => {
    const fullStars = Math.floor(rating);
    const hasHalfStar = rating % 1 >= 0.5;
    const emptyStars = 5 - fullStars - (hasHalfStar ? 1 : 0);

    return (
      <div className="flex items-center gap-0.5">
        {[...Array(fullStars)].map((_, i) => (
          <span key={i} className="text-yellow-400 text-lg leading-none">
            ★
          </span>
        ))}
        {hasHalfStar && (
          <span className="text-yellow-400 text-lg leading-none">★</span>
        )}
        {[...Array(emptyStars)].map((_, i) => (
          <span
            key={i}
            className="text-yellow-400 text-lg opacity-30 leading-none"
          >
            ★
          </span>
        ))}
      </div>
    );
  };

  return (
    <div className="hidden lg:block sticky top-28 bg-white border border-gray-100 rounded-lg p-6 shadow-md w-[35%]" data-booking-card>
      {/* Show salon name and rating only when header is out of view */}
      <div
        className={`overflow-hidden transition-all duration-300 ease-in-out ${
          showHeaderInfo
            ? "max-h-40 opacity-100 mb-3"
            : "max-h-0 opacity-0 mb-0"
        }`}
      >
        <h1 className="heading-1 mb-3">{salon.salonName || "Salon"}</h1>
        {rating > 0 && (
          <div className="mb-3 flex items-center gap-1.5">
            <span className="text-lg font-semibold text-gray-900">
              {rating.toFixed(1)}
            </span>
            <StarDisplay rating={rating} />
            <span className="text-base text-gray-900">({reviewCount})</span>
          </div>
        )}
      </div>

      {/* Deals tag - can be added if deals exist */}
      {/* <div className="mb-3">
        <span className="px-2 py-1 bg-green-100 text-green-700 text-xs font-medium rounded-full">
          Deals
        </span>
      </div> */}

      <Button
        variant="primary"
        size="lg"
        className="w-full mb-4"
        onClick={onBookClick}
      >
        Book now
      </Button>

      <div className="border-t border-gray-200 pt-4 space-y-4">
        {isOpen && openUntil && (
          <div>
            <button
              onClick={() => setShowAllHours(!showAllHours)}
              className="w-full flex items-center gap-2 hover:bg-gray-50 -mx-2 px-2 py-1.5 rounded transition-colors"
            >
              <Clock className="w-4 h-4 text-gray-600 flex-shrink-0" />
              <span className="text-sm text-green-600 flex-1 text-left">
                Open until {openUntil}
              </span>
              {showAllHours ? (
                <ChevronUp className="w-4 h-4 text-gray-400 flex-shrink-0" />
              ) : (
                <ChevronDown className="w-4 h-4 text-gray-400 flex-shrink-0" />
              )}
            </button>

            {/* Expanded opening hours - using same component as SalonOpeningTimes */}
            <div
              className={`overflow-hidden transition-all duration-300 ease-in-out ${
                showAllHours
                  ? "max-h-[500px] opacity-100 mt-3 pt-3 border-t border-gray-100"
                  : "max-h-0 opacity-0 mt-0 pt-0 border-t-0"
              }`}
            >
              <div className="space-y-2">
                {sortedTimes.map((time) => {
                  const isOpen =
                    time.status && time.openingTime && time.closingTime;
                  const hours = isOpen
                    ? `${formatTime(time.openingTime)} - ${formatTime(
                        time.closingTime
                      )}`
                    : "Closed";
                  const isToday = time.day === today;
                  const isClosed = !isOpen;

                  return (
                    <div
                      key={time.day}
                      className="flex items-center gap-3 py-2 border-b border-gray-100 last:border-0"
                    >
                      {/* Green dot for open days, gray dot for closed days */}
                      {isOpen ? (
                        <div className="w-2 h-2 bg-green-500 rounded-full flex-shrink-0" />
                      ) : (
                        <div className="w-2 h-2 bg-gray-400 rounded-full flex-shrink-0" />
                      )}
                      <span
                        className={`text-sm flex-1 ${
                          isToday
                            ? "font-bold text-gray-900"
                            : isClosed
                            ? "text-gray-400 font-medium"
                            : "font-medium text-gray-900"
                        }`}
                      >
                        {time.day}
                      </span>
                      <span
                        className={`text-sm ${
                          isToday
                            ? "font-bold text-gray-900"
                            : isClosed
                            ? "text-gray-400 font-medium"
                            : "text-gray-600 font-medium"
                        }`}
                      >
                        {hours}
                      </span>
                    </div>
                  );
                })}
              </div>
            </div>
          </div>
        )}

        {fullAddress && (
          <div className="flex items-start gap-2">
            <MapPin className="w-4 h-4 text-gray-600 mt-0.5 flex-shrink-0" />
            <div className="flex-1 min-w-0">
              <p className="text-sm text-gray-700 leading-relaxed">
                {fullAddress}
              </p>
              <button className="text-sm text-purple-600 hover:underline mt-1">
                Get directions
              </button>
            </div>
          </div>
        )}
      </div>
    </div>
  );
}
