"use client";

import Image from "next/image";
import { X } from "lucide-react";
import { Button } from "@/components/ui/button";
import { StarRating } from "@/components/data-display/StarRating";
import { PlaceholderImage } from "@/components/data-display/PlaceholderImage";
import { SalonDetail, Employee } from "@/lib/schemas/salonDetail";
import { getImageUrl } from "../utils/imageUtils";

interface SelectedService {
  id?: number;
  serviceName?: string | null;
  duration?: number | null;
  originalPrice?: number;
  discountedPrice?: number;
  discount?: number | null;
}

interface BookingSummaryProps {
  salon: SalonDetail;
  selectedServices: SelectedService[];
  currencyUnit?: string;
  selectedProfessional?: string | null;
  employees?: Employee[];
  onRemoveService?: (serviceId?: number) => void;
  onContinue: () => void;
  continueButtonDisabled?: boolean;
  continueButtonText?: string;
  showPriceBreakdown?: boolean;
}

export function BookingSummary({
  salon,
  selectedServices,
  currencyUnit = "£",
  selectedProfessional,
  employees = [],
  onRemoveService,
  onContinue,
  continueButtonDisabled = false,
  continueButtonText = "Continue",
  showPriceBreakdown = false,
}: BookingSummaryProps) {
  const address = salon.addressDB;
  const fullAddress = [
    address?.building,
    address?.streetAddress,
    address?.city,
    address?.district,
    address?.province,
    address?.country,
  ]
    .filter(Boolean)
    .join(", ");

  const rating = salon.salonAverageRating ?? 0;
  const reviewCount = salon.ratingCount ?? 0;

  const formatDuration = (minutes?: number | null): string => {
    if (!minutes) return "";
    if (minutes < 60) return `${minutes} mins`;
    const hours = Math.floor(minutes / 60);
    const mins = minutes % 60;
    if (mins === 0) return `${hours} hr${hours > 1 ? "s" : ""}`;
    return `${hours} hr ${mins} mins`;
  };

  const getPrice = (service: SelectedService): { original: number; final: number } => {
    const original = service.originalPrice || 0;
    const final = service.discountedPrice || service.originalPrice || 0;
    return { original, final };
  };

  const subtotal = selectedServices.reduce(
    (sum, service) => sum + (service.originalPrice || 0),
    0
  );
  const totalDiscount = selectedServices.reduce(
    (sum, service) => {
      const { original, final } = getPrice(service);
      return sum + (original - final);
    },
    0
  );
  const total = selectedServices.reduce(
    (sum, service) => {
      const { final } = getPrice(service);
      return sum + final;
    },
    0
  );

  const getProfessionalText = () => {
    // Only show professional text if a specific employee is selected
    if (selectedProfessional && selectedProfessional.startsWith("employee-")) {
      // Extract employee ID from "employee-{id}" format
      const employeeId = selectedProfessional.replace("employee-", "");
      const employee = employees.find(
        (emp) => emp.id?.toString() === employeeId
      );
      if (employee) {
        const firstName = employee.user?.firstName || "";
        const lastName = employee.user?.lastName || "";
        const fullName = `${firstName} ${lastName}`.trim();
        if (fullName) {
          return `with ${fullName}`;
        }
      }
    }
    // Return empty string if no professional selected or "any" is selected
    return "";
  };

  return (
    <div className="sticky top-4 bg-white border border-gray-200 rounded-lg p-6 shadow-md">
      {/* Salon Info */}
      <div className="flex items-start gap-3 mb-4">
        {salon.coverImage ? (
          <div className="relative w-16 h-16 rounded-lg overflow-hidden flex-shrink-0">
            <Image
              src={getImageUrl(salon.coverImage)}
              alt={salon.salonName || "Salon"}
              fill
              className="object-cover"
            />
          </div>
        ) : (
          <div className="w-16 h-16 rounded-lg overflow-hidden flex-shrink-0">
            <PlaceholderImage className="w-full h-full" />
          </div>
        )}
        <div className="flex-1 min-w-0">
          <h3 className="font-bold text-gray-900 mb-1">
            {salon.salonName || "Salon"}
          </h3>
          {rating > 0 && (
            <div className="mb-1">
              <StarRating rating={rating} reviewCount={reviewCount} />
            </div>
          )}
          {fullAddress && (
            <p className="text-xs text-gray-600 line-clamp-2">{fullAddress}</p>
          )}
        </div>
      </div>

      <div className="border-t border-gray-200 pt-4 space-y-4">
        {/* Selected Services */}
        {selectedServices.length === 0 ? (
          <p className="text-sm text-gray-500 text-center py-4">
            No services selected
          </p>
        ) : (
          <div className="space-y-3">
            {selectedServices.map((service) => {
              const duration = formatDuration(service.duration);
              const { original, final } = getPrice(service);
              const hasDiscount = original > final;
              const professionalText = getProfessionalText();

              return (
                <div key={service.id} className="space-y-2">
                  <div className="flex items-start justify-between">
                    <div className="flex-1 min-w-0">
                      <p className="font-medium text-gray-900 text-sm">
                        {service.serviceName || "Service"}
                      </p>
                      {duration && (
                        <p className="text-xs text-gray-500">
                          {duration}
                          {professionalText && ` ${professionalText}`}
                        </p>
                      )}
                    </div>
                    {onRemoveService ? (
                      <button
                        onClick={() => onRemoveService(service.id)}
                        className="ml-2 p-1 hover:bg-gray-100 rounded transition-colors"
                      >
                        <X className="w-4 h-4 text-gray-500" />
                      </button>
                    ) : (
                      <p className="text-sm font-semibold text-gray-900 ml-2">
                        {currencyUnit}
                        {final.toFixed(2)}
                      </p>
                    )}
                  </div>
                  {onRemoveService && (
                    <div className="text-right">
                      <p className="text-sm font-semibold text-gray-900">
                        {currencyUnit}
                        {final.toFixed(2)}
                      </p>
                      {hasDiscount && (
                        <p className="text-xs text-gray-400 line-through">
                          {currencyUnit}
                          {original.toFixed(2)}
                        </p>
                      )}
                    </div>
                  )}
                </div>
              );
            })}
          </div>
        )}

        {/* Price Breakdown */}
        {showPriceBreakdown && selectedServices.length > 0 && (
          <div className="border-t border-gray-200 pt-3 space-y-2">
            <div className="flex justify-between text-sm">
              <span className="text-gray-600">Subtotal</span>
              <span className="text-gray-900">
                {currencyUnit}
                {subtotal.toFixed(2)}
              </span>
            </div>
            {totalDiscount > 0 && (
              <div className="flex justify-between text-sm">
                <span className="text-gray-600">Discounts</span>
                <span className="text-green-600">
                  -{currencyUnit}
                  {totalDiscount.toFixed(2)}
                </span>
              </div>
            )}
          </div>
        )}

        {/* Total */}
        {selectedServices.length > 0 && (
          <div className="border-t border-gray-200 pt-3">
            <div className="flex justify-between items-center">
              <span className="font-bold text-gray-900">Total</span>
              <span className="font-bold text-gray-900">
                {currencyUnit}
                {total.toFixed(2)}
              </span>
            </div>
          </div>
        )}

        {/* Continue Button */}
        <Button
          variant="primary"
          size="lg"
          className="w-full mt-4"
          disabled={continueButtonDisabled}
          onClick={onContinue}
        >
          {continueButtonText}
        </Button>
      </div>
    </div>
  );
}

