/**
 * Transform FavoriteVenue from API to display format
 */

import { FavoriteVenue } from "@/lib/schemas/favorites";
import { API_BASE_URL } from "@/config/env";

export interface FavoriteDisplay {
  id: string;
  name: string;
  location: string;
  rating: number;
  reviewCount: number;
  service: string;
  image: string;
}

export function transformFavoriteToDisplay(favorite: FavoriteVenue): FavoriteDisplay {
  // Build location string
  const address = favorite.addressDB;
  const location = address
    ? `${address.district || address.city}, ${address.city}`
    : "Location not available";

  // Build service string from services array
  const services = favorite.services || [];
  const service = services.length > 0
    ? services.map((s) => s.serviceName).join(", ")
    : "Services";

  // Handle image URL - convert relative paths to absolute URLs
  // Use placeholder URL that can be detected for showing placeholder component
  let imageUrl = "/placeholder-venue";
  if (favorite.coverImage) {
    if (favorite.coverImage.startsWith("http://") || favorite.coverImage.startsWith("https://")) {
      imageUrl = favorite.coverImage;
    } else if (favorite.coverImage.startsWith("/")) {
      imageUrl = `${API_BASE_URL}${favorite.coverImage}`;
    } else {
      // Relative path like "Public/Gallery/..." - prepend base URL with leading slash
      imageUrl = `${API_BASE_URL}/${favorite.coverImage}`;
    }
  }

  return {
    id: favorite.id.toString(),
    name: favorite.salonName,
    location,
    rating: favorite.salonAverageRating ?? 0,
    reviewCount: favorite.ratingCount,
    service,
    image: imageUrl,
  };
}

