"use client";

import { useState } from "react";
import Image from "next/image";
import { Edit, Pencil } from "lucide-react";
import { Button } from "@/components/ui/button";
import { getUserData } from "@/lib/auth";
import { API_BASE_URL } from "@/config/env";

export function ProfileDetailsSection() {
  const [imageError, setImageError] = useState(false);
  const userData = getUserData();

  const getFirstLetter = () => {
    if (userData?.firstName) {
      return userData.firstName.charAt(0).toUpperCase();
    }
    if (userData?.email) {
      return userData.email.charAt(0).toUpperCase();
    }
    return "U";
  };

  const hasProfileImage = () => {
    return userData?.image && userData.image.trim() !== "" && !imageError;
  };

  const getProfileImageUrl = () => {
    if (!userData?.image) return "";
    const imageUrl = userData.image.trim();
    if (imageUrl.startsWith("http://") || imageUrl.startsWith("https://")) {
      return imageUrl;
    }
    // Use API_BASE_URL for relative paths
    const baseUrl = API_BASE_URL.endsWith("/") 
      ? API_BASE_URL.slice(0, -1) 
      : API_BASE_URL;
    return imageUrl.startsWith("/")
      ? `${baseUrl}${imageUrl}`
      : imageUrl;
  };

  const fullName =
    userData?.firstName && userData?.lastName
      ? `${userData.firstName} ${userData.lastName}`
      : userData?.email || "User";

  return (
    <div className="bg-white rounded-xl border border-gray-200 p-4 md:p-6">
      {/* Edit Button - Top Right */}
      <div className="flex justify-end mb-4 lg:hidden">
        <button className="text-sm md:text-base text-purple-600 font-medium hover:text-purple-700 transition-colors">
          Edit
        </button>
      </div>
      <div className="hidden lg:flex items-start justify-between mb-6">
        <h1 className="heading-4 text-gray-900">Profile</h1>
        <Button variant="outline" size="sm">
          <Edit className="w-4 h-4 mr-2" />
          Edit
        </Button>
      </div>

      <div className="flex flex-col items-center mb-6">
        <div className="relative mb-4">
          {hasProfileImage() ? (
            <div className="w-20 h-20 md:w-24 md:h-24 rounded-full overflow-hidden relative">
              <Image
                src={getProfileImageUrl()}
                alt={`${userData?.firstName || "User"} profile`}
                fill
                className="object-cover"
                onError={() => setImageError(true)}
              />
            </div>
          ) : (
            <div className="w-20 h-20 md:w-24 md:h-24 rounded-full bg-teal-500 flex items-center justify-center text-white font-semibold text-2xl md:text-3xl">
              {getFirstLetter()}
            </div>
          )}
          <button className="absolute bottom-0 right-0 w-7 h-7 md:w-8 md:h-8 bg-white rounded-full border-2 border-white flex items-center justify-center shadow-md hover:bg-gray-50 transition-colors">
            <Pencil className="w-3 h-3 md:w-4 md:h-4 text-gray-700" />
          </button>
        </div>
        <h2 className="text-lg md:text-xl font-bold text-gray-900">{fullName}</h2>
      </div>

      <div className="border-t border-gray-200 pt-4 md:pt-6">
        <div className="space-y-4 md:space-y-5">
          <div>
            <label className="text-xs md:text-sm font-semibold text-gray-900 mb-1 block">First name</label>
            <p className="text-sm md:text-base text-gray-600">
              {userData?.firstName || "-"}
            </p>
          </div>

          <div>
            <label className="text-xs md:text-sm font-semibold text-gray-900 mb-1 block">Last name</label>
            <p className="text-sm md:text-base text-gray-600">
              {userData?.lastName || "-"}
            </p>
          </div>

          <div>
            <label className="text-xs md:text-sm font-semibold text-gray-900 mb-1 block">
              Mobile number
            </label>
            <p className="text-sm md:text-base text-gray-600">
              {userData?.phoneNum || "-"}
            </p>
          </div>

          <div>
            <label className="text-xs md:text-sm font-semibold text-gray-900 mb-1 block">Email</label>
            <p className="text-sm md:text-base text-gray-600">{userData?.email || "-"}</p>
          </div>

          <div>
            <label className="text-xs md:text-sm font-semibold text-gray-900 mb-1 block">
              Date of birth
            </label>
            <p className="text-sm md:text-base text-gray-600">-</p>
          </div>

          <div>
            <label className="text-xs md:text-sm font-semibold text-gray-900 mb-1 block">Gender</label>
            <p className="text-sm md:text-base text-gray-600">-</p>
          </div>
        </div>
      </div>
    </div>
  );
}
