/**
 * Server-side: Get favorites
 * Use in Server Components, Server Actions, Route Handlers
 */

import "server-only";
import {
  GetFavoritesRequestSchema,
  GetFavoritesResponseSchema,
  type GetFavoritesRequest,
  type GetFavoritesResponse,
} from "@/lib/schemas/favorites";

export async function getFavoritesServer(
  request: GetFavoritesRequest
): Promise<GetFavoritesResponse> {
  // Validate request
  const validatedRequest = GetFavoritesRequestSchema.parse(request);

  // Build query parameters
  const queryParams = new URLSearchParams();
  queryParams.set("targetLat", validatedRequest.targetLat.toString());
  queryParams.set("targetLng", validatedRequest.targetLng.toString());
  queryParams.set("rangeInKm", validatedRequest.rangeInKm.toString());

  const endpoint = `/customer/favorites/salons?${queryParams.toString()}`;

  // Dynamically import server fetcher to avoid bundling in client code
  const { serverFetcher } = await import("@/lib/api/fetcher-server");

  const data = await serverFetcher<unknown>(endpoint, {
    requireAuth: true, // Requires authentication
    method: "GET",
  });

  // Parse with better error handling
  try {
    return GetFavoritesResponseSchema.parse(data);
  } catch (error) {
    // Log the raw data for debugging
    console.error("Zod validation failed. Raw data:", JSON.stringify(data, null, 2));
    throw error;
  }
}

