/**
 * Cancel Appointment API - Client Side
 * Cancels an appointment with cancellation charges
 */

"use client";

import { clientFetcher } from "@/lib/api/fetcher-client";
import { z } from "zod";

export const CancelAppointmentRequestSchema = z.object({
  cancellationCharges: z.number(),
});

export type CancelAppointmentRequest = z.infer<
  typeof CancelAppointmentRequestSchema
>;

/**
 * Client-side: Cancel appointment
 * Use in Client Components, React Query hooks
 */
export async function cancelAppointmentClient(
  appointmentId: number,
  request: CancelAppointmentRequest
): Promise<unknown> {
  // Validate request
  const validatedRequest = CancelAppointmentRequestSchema.parse(request);

  // Log the request for debugging
  console.log("Cancel appointment request:", {
    appointmentId,
    endpoint: `/customer/cancel-appointment/${appointmentId}`,
    body: validatedRequest,
  });

  try {
    const endpoint = `/customer/cancel-appointment/${appointmentId}`;
    console.log("Cancel appointment - Making request to:", endpoint);
    console.log("Cancel appointment - Request body:", validatedRequest);
    
    const data = await clientFetcher<unknown>(
      endpoint,
      {
        requireAuth: true,
        method: "PATCH",
        body: JSON.stringify(validatedRequest),
      }
    );

    console.log("Cancel appointment - Success:", data);
    return data;
  } catch (error: any) {
    console.error("Cancel appointment error:", error);
    console.error("Cancel appointment error details:", {
      message: error?.message,
      status: error?.status,
      statusCode: error?.statusCode,
      error: error?.error,
      stack: error?.stack,
    });
    
    // Re-throw with more context
    const errorMessage = 
      error?.message || 
      error?.error || 
      `Failed to cancel appointment. ${error?.statusCode ? `HTTP ${error.statusCode}` : ""}`;
    
    throw new Error(errorMessage);
  }
}

