"use client";

import { useState } from "react";
import { useRouter } from "next/navigation";
import Image from "next/image";
import { Card } from "@/components/ui/card";
import { StarRating } from "@/components/data-display/StarRating";
import { Venue } from "@/types/venue";
import { PlaceholderImage } from "@/components/data-display/PlaceholderImage";

interface VenueCardProps {
  venue: Venue;
  fullWidth?: boolean;
}

export function VenueCard({ venue, fullWidth = false }: VenueCardProps) {
  const [imageError, setImageError] = useState(false);
  const router = useRouter();

  const handleClick = () => {
    router.push(`/saloons/${venue.id}`);
  };

  return (
    <div className="h-full w-full">
      <Card
        hover
        className={`flex-shrink-0 ${fullWidth ? "w-full" : "w-[280px] sm:w-[300px] md:w-[320px] lg:w-card"} h-full flex flex-col min-h-[320px] sm:min-h-[340px] md:min-h-card transition-transform cursor-pointer`}
        style={{
          transform: "translateY(0)",
        }}
        onClick={handleClick}
        onMouseEnter={(e: React.MouseEvent<HTMLDivElement>) => {
          e.currentTarget.style.transform =
            "translateY(var(--card-hover-translate))";
        }}
        onMouseLeave={(e: React.MouseEvent<HTMLDivElement>) => {
          e.currentTarget.style.transform = "translateY(0)";
        }}
      >
        {/* Image Area - Full width, fixed height */}
        <div
          className="relative w-full flex-shrink-0 overflow-hidden h-[180px] sm:h-[200px] md:h-[220px] lg:h-[var(--card-image-height)]"
          style={{
            borderTopLeftRadius: "var(--card-border-radius)",
            borderTopRightRadius: "var(--card-border-radius)",
          }}
        >
          {imageError || !venue.image || venue.image.includes("placeholder") ? (
            <PlaceholderImage
              className="w-full h-full"
              style={{
                borderTopLeftRadius: "var(--card-border-radius)",
                borderTopRightRadius: "var(--card-border-radius)",
                borderBottomLeftRadius: "var(--card-border-radius)",
                borderBottomRightRadius: "var(--card-border-radius)",
              }}
            />
          ) : (
            <Image
              src={venue.image}
              alt={venue.name}
              fill
              className="object-cover"
              style={{
                borderTopLeftRadius: "var(--card-border-radius)",
                borderTopRightRadius: "var(--card-border-radius)",
                borderBottomLeftRadius: "var(--card-border-radius)",
                borderBottomRightRadius: "var(--card-border-radius)",
              }}
              onError={() => setImageError(true)}
            />
          )}

          {/* Optional Badge - Top-left corner */}
          {venue.badge && (
            <div className="absolute top-2 left-2 px-2 py-0.5 bg-white/90 backdrop-blur-sm rounded-full">
              <span
                className="text-xs font-bold"
                style={{ color: "var(--text-primary)" }}
              >
                {venue.badge}
              </span>
            </div>
          )}
        </div>

        {/* Text Block - Consistent padding from theme */}
        <div className="flex flex-col flex-1 py-3 sm:py-4">
          {/* Row 1: Name (left) + Rating (right) */}
          <div className="flex items-start justify-between gap-2 mb-1.5">
            {/* Name - 1 line clamp, larger and bolder */}
            <h3
              className="text-base sm:text-lg font-bold flex-1 min-w-0 line-clamp-1"
              style={{ color: "var(--text-primary)" }}
            >
              {venue.name}
            </h3>

            {/* Rating - Right-aligned, single line */}
            <div className="flex-shrink-0">
              <StarRating
                rating={venue.rating}
                reviewCount={venue.reviewCount}
              />
            </div>
          </div>

          {/* Row 2: Address - 1 line clamp */}
          <p
            className="text-xs sm:text-sm mb-1.5 line-clamp-1 px-0 font-normal"
            style={{ color: "var(--text-secondary)" }}
          >
            {venue.address || venue.location}
          </p>

          {/* Row 3: Category */}
          <p
            className="text-xs sm:text-sm mt-2 font-normal"
            style={{ color: "var(--text-muted)" }}
          >
            {venue.category}
          </p>
        </div>
      </Card>
    </div>
  );
}
