"use client";

import { Star, ChevronLeft, ChevronRight } from "lucide-react";
import { useRef } from "react";
import { Container } from "@/components/layout/Container";
import { PageSection } from "@/components/layout/PageSection";
import { Card } from "@/components/ui/card";

interface Review {
  rating: number;
  title: string;
  body: string;
  reviewerName: string;
  location: string;
}

const reviews: Review[] = [
  {
    rating: 5,
    title: "The best booking system",
    body: "Great experience, easy to book. Paying for treatments is so convenient — no cash or cards needed!",
    reviewerName: "Lucy",
    location: "London, UK",
  },
  {
    rating: 5,
    title: "Easy to use & explore",
    body: "Fresha's premium make life so much easier! I also found a few good barbershops that I didn't know existed.",
    reviewerName: "Dan",
    location: "New York, USA",
  },
  {
    rating: 5,
    title: "Great for finding barbers",
    body: "I've been using Fresha for two years and it's by far the best booking platform I've used. Highly recommend it!",
    reviewerName: "Dale",
    location: "Sydney, Australia",
  },
  {
    rating: 5,
    title: "My go-to for self-care",
    body: "Fresha is my go-to app for haircuts and facials. I can easily find and book places near me — I love it!",
    reviewerName: "Cameron",
    location: "Edinburgh, UK",
  },
];

export function ReviewsSection() {
  const scrollRef = useRef<HTMLDivElement>(null);

  const scroll = (direction: "left" | "right") => {
    if (scrollRef.current) {
      const scrollAmount = 400;
      scrollRef.current.scrollBy({
        left: direction === "left" ? -scrollAmount : scrollAmount,
        behavior: "smooth",
      });
    }
  };

  return (
    <PageSection>
      <Container>
        <div className="py-8 sm:py-12 md:py-16">
          <div className="mb-6 sm:mb-8">
            <h2
              className="text-2xl sm:text-3xl md:text-4xl font-bold"
              style={{ color: "var(--text-primary)" }}
            >
              Reviews
            </h2>
          </div>

          <div className="relative">
            {/* Left Arrow - Hidden on mobile, shown on tablet+ */}
            <button
              onClick={() => scroll("left")}
              className="absolute left-0 top-1/2 -translate-y-1/2 z-10 rounded-full p-2 md:p-2.5 lg:p-3 transition-shadow hidden md:block"
              style={{
                backgroundColor: "var(--surface)",
                border: "1px solid var(--color-gray-300)",
                boxShadow: "var(--shadow-sm)",
              }}
              aria-label="Scroll left"
            >
              <ChevronLeft
                className="w-4 h-4 md:w-5 md:h-5"
                style={{ color: "var(--text-primary)" }}
              />
            </button>

            {/* Review Cards Container - Equal heights */}
            <div
              ref={scrollRef}
              className="flex gap-3 sm:gap-4 md:gap-6 overflow-x-auto scrollbar-hide scroll-smooth pb-4 items-stretch px-2 md:px-0"
              style={{
                scrollbarWidth: "none",
                msOverflowStyle: "none",
                WebkitOverflowScrolling: "touch",
              }}
            >
              {reviews.map((review, index) => (
                <div
                  key={index}
                  className="flex-shrink-0 w-[260px] sm:w-[280px] md:w-[320px] lg:w-[340px] flex flex-col p-5 sm:p-6 md:p-7 rounded-2xl"
                  style={{
                    backgroundColor: "#f5f5f5",
                  }}
                >
                  {/* Star Row - Golden-yellow */}
                  <div className="flex gap-1 mb-3 sm:mb-4">
                    {[...Array(review.rating)].map((_, i) => (
                      <Star
                        key={i}
                        className="w-4 h-4"
                        style={{
                          fill: "#facc15",
                          color: "#facc15",
                        }}
                      />
                    ))}
                  </div>

                  <h3
                    className="text-lg sm:text-xl font-bold mb-3 line-clamp-2"
                    style={{ color: "#000000" }}
                  >
                    {review.title}
                  </h3>

                  <p
                    className="text-sm sm:text-base mb-6 sm:mb-8 line-clamp-3"
                    style={{ color: "#4a5568" }}
                  >
                    {review.body}
                  </p>

                  <div className="flex items-center gap-3 mt-auto">
                    <div className="w-11 h-11 rounded-full flex items-center justify-center text-white text-base font-semibold flex-shrink-0" style={{ backgroundColor: "#10b981" }}>
                      {review.reviewerName.charAt(0)}
                    </div>
                    <div>
                      <p
                        className="text-base font-semibold"
                        style={{ color: "#000000" }}
                      >
                        {review.reviewerName}
                      </p>
                      <p
                        className="text-sm"
                        style={{ color: "#6b7280" }}
                      >
                        {review.location}
                      </p>
                    </div>
                  </div>
                </div>
              ))}
            </div>

            {/* Right Arrow - Hidden on mobile, shown on tablet+ */}
            <button
              onClick={() => scroll("right")}
              className="absolute right-0 top-1/2 -translate-y-1/2 z-10 rounded-full p-2 md:p-2.5 lg:p-3 transition-shadow hidden md:block"
              style={{
                backgroundColor: "var(--surface)",
                border: "1px solid var(--color-gray-300)",
                boxShadow: "var(--shadow-sm)",
              }}
              aria-label="Scroll right"
            >
              <ChevronRight
                className="w-4 h-4 md:w-5 md:h-5"
                style={{ color: "var(--text-primary)" }}
              />
            </button>
          </div>
        </div>
      </Container>
    </PageSection>
  );
}
