/**
 * Get Available Slots API
 * Fetches available time slots for booking
 */

import { clientFetcher } from "@/lib/api/fetcher-client";
import {
  TimeSlotSchema,
  AvailableSlotsRequestSchema,
  type TimeSlot,
  type AvailableSlotsRequest,
} from "@/lib/schemas/booking";
import { z } from "zod";

const AvailableSlotsResponseSchema = z.array(TimeSlotSchema);

/**
 * Server-side: Get available slots
 * Dynamically imports server fetcher to avoid bundling in client code
 */
export async function getAvailableSlotsServer(
  request: AvailableSlotsRequest
): Promise<TimeSlot[]> {
  // Validate request
  const validatedRequest = AvailableSlotsRequestSchema.parse(request);

  // Dynamically import server fetcher to avoid bundling in client
  const { serverFetcher } = await import("@/lib/api/fetcher-server");

  const queryParams = new URLSearchParams();
  queryParams.set("venueId", validatedRequest.venueId);
  queryParams.set("date", validatedRequest.date);
  queryParams.set("duration", validatedRequest.duration.toString());
  validatedRequest.serviceIds.forEach((id) => {
    queryParams.append("serviceIds", id);
  });
  if (validatedRequest.professionalId) {
    queryParams.set("professionalId", validatedRequest.professionalId);
  }

  const endpoint = `/bookings/available-slots?${queryParams.toString()}`;

  const data = await serverFetcher<unknown>(endpoint, {
    requireAuth: false, // Public endpoint
    method: "GET",
  });

  return AvailableSlotsResponseSchema.parse(data);
}

/**
 * Client-side: Get available slots
 */
export async function getAvailableSlotsClient(
  request: AvailableSlotsRequest
): Promise<TimeSlot[]> {
  // Validate request
  const validatedRequest = AvailableSlotsRequestSchema.parse(request);

  const queryParams = new URLSearchParams();
  queryParams.set("venueId", validatedRequest.venueId);
  queryParams.set("date", validatedRequest.date);
  queryParams.set("duration", validatedRequest.duration.toString());
  validatedRequest.serviceIds.forEach((id) => {
    queryParams.append("serviceIds", id);
  });
  if (validatedRequest.professionalId) {
    queryParams.set("professionalId", validatedRequest.professionalId);
  }

  const endpoint = `/bookings/available-slots?${queryParams.toString()}`;

  const data = await clientFetcher<unknown>(endpoint, {
    requireAuth: false,
    method: "GET",
  });

  return AvailableSlotsResponseSchema.parse(data);
}
