/**
 * Create Booking API
 * Creates a new booking
 */

import { clientFetcher } from "@/lib/api/fetcher-client";
import {
  BookingSchema,
  CreateBookingRequestSchema,
  type Booking,
  type CreateBookingRequest,
} from "@/lib/schemas/booking";

/**
 * Server-side: Create booking
 * Dynamically imports server fetcher to avoid bundling in client code
 */
export async function createBookingServer(
  request: CreateBookingRequest
): Promise<Booking> {
  // Validate request
  const validatedRequest = CreateBookingRequestSchema.parse(request);

  // Dynamically import server fetcher to avoid bundling in client
  const { serverFetcher } = await import("@/lib/api/fetcher-server");

  const data = await serverFetcher<unknown>("/bookings", {
    requireAuth: true,
    method: "POST",
    body: JSON.stringify(validatedRequest),
  });

  return BookingSchema.parse(data);
}

/**
 * Client-side: Create booking
 */
export async function createBookingClient(
  request: CreateBookingRequest
): Promise<Booking> {
  // Validate request
  const validatedRequest = CreateBookingRequestSchema.parse(request);

  const data = await clientFetcher<unknown>("/bookings", {
    requireAuth: true,
    method: "POST",
    body: JSON.stringify(validatedRequest),
  });

  return BookingSchema.parse(data);
}
