/**
 * Social Authentication API
 * Handles Google and Apple authentication via Firebase
 */

import {
  signInWithPopup,
  GoogleAuthProvider,
  OAuthProvider,
  User,
} from "firebase/auth";
import { auth } from "@/lib/firebase";
import { signup } from "./signup";
import { checkEmail } from "./emailChecker";
import { setAccessToken, setUserData } from "@/lib/auth";

export type SocialProvider = "google" | "apple";

export interface SocialAuthResult {
  success: boolean;
  user?: User;
  error?: string;
  requiresSignup?: boolean;
}

/**
 * Sign in with Google
 */
export async function signInWithGoogle(): Promise<SocialAuthResult> {
  try {
    const provider = new GoogleAuthProvider();
    const result = await signInWithPopup(auth, provider);
    const user = result.user;

    return {
      success: true,
      user,
    };
  } catch (error: any) {
    console.error("Google sign-in error:", error);
    return {
      success: false,
      error: error.message || "Failed to sign in with Google",
    };
  }
}

/**
 * Sign in with Apple
 */
export async function signInWithApple(): Promise<SocialAuthResult> {
  try {
    const provider = new OAuthProvider("apple.com");
    const result = await signInWithPopup(auth, provider);
    const user = result.user;

    return {
      success: true,
      user,
    };
  } catch (error: any) {
    console.error("Apple sign-in error:", error);
    return {
      success: false,
      error: error.message || "Failed to sign in with Apple",
    };
  }
}

/**
 * Check if social auth data is complete
 */
function isSocialAuthDataComplete(user: User): boolean {
  // Always return false to redirect to signup form
  // This ensures we collect phone number and verify all data
  // Social providers rarely provide phone numbers, so we need to collect it
  return false;
}

/**
 * Extract user data from Firebase user
 */
export function extractSocialAuthData(user: User) {
  const email = user.email || "";
  const displayName = user.displayName || "";
  const nameParts = displayName.split(" ");
  const firstName = nameParts[0] || "";
  const lastName = nameParts.slice(1).join(" ") || "";
  const phoneNumber = user.phoneNumber || "";
  const image = user.photoURL || "";

  return {
    email,
    firstName,
    lastName,
    phoneNumber,
    image,
  };
}

/**
 * Handle social authentication flow
 * Checks if user exists, creates account if needed, or logs in
 * Returns user data if incomplete to pre-fill signup form
 */
export async function handleSocialAuth(
  provider: SocialProvider
): Promise<{ 
  success: boolean; 
  error?: string; 
  requiresOtp?: boolean;
  requiresSignup?: boolean;
  userData?: {
    email: string;
    firstName: string;
    lastName: string;
    phoneNumber: string;
    image: string;
    signedBy: "google" | "apple";
  };
}> {
  try {
    // Sign in with Firebase
    let firebaseUser: User;
    if (provider === "google") {
      const result = await signInWithGoogle();
      if (!result.success || !result.user) {
        return { success: false, error: result.error || "Authentication failed" };
      }
      firebaseUser = result.user;
    } else {
      const result = await signInWithApple();
      if (!result.success || !result.user) {
        return { success: false, error: result.error || "Authentication failed" };
      }
      firebaseUser = result.user;
    }

    // Extract user data from Firebase
    const email = firebaseUser.email;
    if (!email) {
      return { success: false, error: "Email not available from social provider" };
    }

    // Extract user data
    const extractedData = extractSocialAuthData(firebaseUser);

    // Check if user exists in our system
    const emailCheck = await checkEmail(email);

    // If user exists and has access token, log them in
    if (emailCheck.status === "1" || emailCheck.status === "2") {
      const userData = emailCheck.data as any;
      if (userData && userData.accessToken) {
        setAccessToken(userData.accessToken);
        setUserData(userData);
        return { success: true };
      }
    }

    // Check if data is complete (has phone number and name)
    const isComplete = isSocialAuthDataComplete(firebaseUser);

    // If data is incomplete, redirect to signup form with pre-filled data
    if (!isComplete) {
      return {
        success: true,
        requiresSignup: true,
        userData: {
          ...extractedData,
          signedBy: provider,
        },
      };
    }

    // Data is complete - create account via signup API
    const signupData = {
      firstName: extractedData.firstName || "User",
      lastName: extractedData.lastName || "",
      email: extractedData.email,
      countryCode: "+1", // Default, user can update later
      phoneNum: extractedData.phoneNumber,
      password: "social-auth-temp-password", // Temporary password for social auth
      image: extractedData.image,
    };

    try {
      const signupResponse = await signup({
        ...signupData,
        signedBy: provider,
      });

      // If signup returns OTP, user needs to verify
      if (signupResponse.otpId && signupResponse.userId) {
        return {
          success: true,
          requiresOtp: true,
        };
      }

      return { success: true };
    } catch (signupError: any) {
      // If signup fails, try to check email again (user might have been created)
      const recheck = await checkEmail(email);
      if (recheck.status === "1" || recheck.status === "2") {
        const userData = recheck.data as any;
        if (userData && userData.accessToken) {
          setAccessToken(userData.accessToken);
          setUserData(userData);
          return { success: true };
        }
      }

      // If signup fails due to incomplete data, redirect to signup form
      return {
        success: true,
        requiresSignup: true,
        userData: {
          ...extractedData,
          signedBy: provider,
        },
      };
    }
  } catch (error: any) {
    console.error("Social auth error:", error);
    return {
      success: false,
      error: error.message || "Authentication failed",
    };
  }
}

