/**
 * Signup API
 * Creates a new customer account
 */

import { clientFetcher } from "@/lib/api/fetcher-client";
import { API_ENDPOINTS } from "@/config/env";
import { getDeviceToken } from "@/lib/deviceToken";

export interface SignupRequest {
  firstName: string;
  lastName: string;
  email: string;
  countryCode: string;
  phoneNum: string;
  password: string;
  signedBy: "email" | "google" | "apple" | "facebook";
  dvToken: string;
  image?: string;
}

export interface SignupResponse {
  otpId?: number;
  userId?: number;
  accessToken?: string;
  // User data might be returned directly for social auth
  [key: string]: any;
}

/**
 * Create a new customer account
 * Returns OTP ID and User ID for verification (email signup)
 * Or access token directly (social auth - if API supports it)
 */
export async function signup(
  data: Omit<SignupRequest, "dvToken"> & { signedBy: "email" | "google" | "apple" | "facebook" }
): Promise<SignupResponse> {
  const dvToken = getDeviceToken();

  const response = await clientFetcher<SignupResponse>(API_ENDPOINTS.signup, {
    requireAuth: false,
    method: "POST",
    body: JSON.stringify({
      ...data,
      dvToken,
      image: data.image || "",
    }),
  });

  return response;
}

