/**
 * Email Checker API
 * Validates email existence and returns user data if exists
 */

import { API_BASE_URL, API_ENDPOINTS } from "@/config/env";
import { getDeviceToken } from "@/lib/deviceToken";

export interface EmailCheckerRequest {
  email: string;
  dvToken: string;
}

export interface UserData {
  accessToken: string;
  userId: string;
  image: string;
  firstName: string;
  lastName: string;
  email: string;
  phoneNum: string;
  stripeCustomerId: string | null;
}

export interface EmailCheckerResponse {
  status: string;
  message: string;
  data: UserData | {};
  error: string;
}

/**
 * Check if email exists in the system
 * Returns user data if exists, empty data if not
 */
export async function checkEmail(
  email: string
): Promise<EmailCheckerResponse> {
  const dvToken = getDeviceToken();

  try {
    // Use raw fetch to get full response with status
    const response = await fetch(`${API_BASE_URL}${API_ENDPOINTS.emailChecker}`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        Accept: "application/json",
      },
      body: JSON.stringify({
        email,
        dvToken,
      }),
    });

    const apiResponse: EmailCheckerResponse = await response.json();

    // Return the full response including status
    return {
      status: apiResponse.status,
      message: apiResponse.message,
      data: apiResponse.data,
      error: apiResponse.error,
    };
  } catch (error: any) {
    // If API returns error status, user doesn't exist
    // Return empty response
    return {
      status: "0",
      message: error.message || "User not found",
      data: {},
      error: error.error || "",
    };
  }
}

