import { InputHTMLAttributes, forwardRef } from "react";
import { cn } from "@/lib/classNames";
import { AlertCircle } from "lucide-react";

interface InputProps extends InputHTMLAttributes<HTMLInputElement> {
  label?: string;
  error?: string;
  touched?: boolean;
}

export const Input = forwardRef<HTMLInputElement, InputProps>(
  ({ label, error, touched, className = "", id, name, ...props }, ref) => {
    const showError = touched && error;
    // Use name as fallback for id if id is not provided, for better automation support
    const inputId = id || name;
    // Use id as name if name is not provided, for better form submission
    const inputName = name || id;
    // Allow data-testid to be overridden from props, otherwise generate one
    const testId = (props as any)['data-testid'] || (inputId ? `input-${inputId}` : undefined);

    return (
      <div className="w-full">
        {label && (
          <label
            htmlFor={inputId}
            className={cn(
              "label block mb-2",
              showError && "text-error-500"
            )}
          >
            {label}
          </label>
        )}
        <div className="relative">
          <input
            ref={ref}
            id={inputId}
            name={inputName}
            data-testid={testId}
            className={cn(
              "w-full px-4 py-3 rounded-xl border transition-colors",
              "body-text",
              showError
                ? "border-error-500 focus:border-error-500 focus:ring-error-500 bg-error-50"
                : "border-gray-300 focus:border-gray-900 focus:ring-gray-900 bg-white",
              "focus:outline-none focus:ring-2 focus:ring-offset-0",
              showError && "pr-10",
              className
            )}
            aria-invalid={showError ? true : undefined}
            aria-describedby={showError ? `${inputId || "input"}-error` : undefined}
            {...props}
          />
          {showError && (
            <AlertCircle className="absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 text-error-500 pointer-events-none" />
          )}
        </div>
        {showError && (
          <p
            id={`${inputId || "input"}-error`}
            className="caption text-error-500 mt-1 flex items-center gap-1"
            role="alert"
          >
            <AlertCircle className="w-3 h-3" />
            {error}
          </p>
        )}
      </div>
    );
  }
);

Input.displayName = "Input";
