"use client";

import { useEffect, useState, useRef } from "react";
import { X, ArrowLeft } from "lucide-react";
import { Button } from "./button";

interface DialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  children: React.ReactNode;
  showCloseButton?: boolean;
  bottomSheet?: boolean; // For mobile bottom sheet style
  showDragHandle?: boolean; // Show drag handle for bottom sheets on mobile
  showBackButton?: boolean; // Show back button
  onBack?: () => void; // Back button handler
}

export function Dialog({
  open,
  onOpenChange,
  children,
  showCloseButton = true,
  bottomSheet = false,
  showDragHandle = true,
  showBackButton = false,
  onBack,
}: DialogProps) {
  const [isVisible, setIsVisible] = useState(false);
  const [shouldRender, setShouldRender] = useState(false);
  const [dragY, setDragY] = useState(0);
  const [isDragging, setIsDragging] = useState(false);
  const sheetRef = useRef<HTMLDivElement>(null);
  const startYRef = useRef(0);
  const currentYRef = useRef(0);

  useEffect(() => {
    if (open) {
      setShouldRender(true);
      // Small delay to ensure DOM is ready before animation
      const timer = setTimeout(() => {
        setIsVisible(true);
      }, 10);
      return () => clearTimeout(timer);
    } else {
      setIsVisible(false);
      setDragY(0);
      setIsDragging(false);
      // Wait for animation to complete before removing from DOM
      const timer = setTimeout(() => {
        setShouldRender(false);
      }, 300); // Match transition duration
      return () => clearTimeout(timer);
    }
  }, [open]);

  // Swipe-to-dismiss gesture handling for mobile bottom sheets
  useEffect(() => {
    if (!bottomSheet || !open || !sheetRef.current) return;

    const sheet = sheetRef.current;
    let startY = 0;
    let currentY = 0;
    let isDragging = false;

    const handleTouchStart = (e: TouchEvent) => {
      // Only allow dragging from the top portion of the sheet (drag handle area)
      const touch = e.touches[0];
      const rect = sheet.getBoundingClientRect();
      const touchY = touch.clientY;
      
      // Allow dragging if touch is within top 60px (drag handle area)
      if (touchY - rect.top < 60) {
        startY = touch.clientY;
        currentY = startY;
        isDragging = true;
        setIsDragging(true);
        sheet.style.transition = "none";
      }
    };

    const handleTouchMove = (e: TouchEvent) => {
      if (!isDragging) return;
      
      currentY = e.touches[0].clientY;
      const deltaY = currentY - startY;
      
      // Only allow downward dragging
      if (deltaY > 0) {
        setDragY(deltaY);
        sheet.style.transform = `translateY(${deltaY}px)`;
      }
    };

    const handleTouchEnd = () => {
      if (!isDragging) return;
      
      isDragging = false;
      setIsDragging(false);
      sheet.style.transition = "";
      
      const deltaY = currentY - startY;
      const threshold = 100; // Minimum drag distance to dismiss
      
      if (deltaY > threshold) {
        // Dismiss the sheet
        onOpenChange(false);
      } else {
        // Snap back
        setDragY(0);
        sheet.style.transform = "";
      }
    };

    sheet.addEventListener("touchstart", handleTouchStart, { passive: true });
    sheet.addEventListener("touchmove", handleTouchMove, { passive: true });
    sheet.addEventListener("touchend", handleTouchEnd, { passive: true });

    return () => {
      sheet.removeEventListener("touchstart", handleTouchStart);
      sheet.removeEventListener("touchmove", handleTouchMove);
      sheet.removeEventListener("touchend", handleTouchEnd);
    };
  }, [bottomSheet, open, onOpenChange]);

  if (!shouldRender) return null;

  return (
    <>
      {/* Dark Overlay/Backdrop */}
      <div
        className={`fixed inset-0 z-[100] ${
          isDragging ? "" : "transition-opacity duration-300"
        } ${isVisible ? "opacity-100" : "opacity-0"}`}
        style={{
          backgroundColor: "rgba(0, 0, 0, 0.4)",
          pointerEvents: isVisible && !isDragging ? "auto" : "none",
          opacity: isDragging && dragY > 0 
            ? Math.max(0, 0.4 * (1 - Math.min(dragY / 400, 1))) 
            : undefined,
        }}
        onClick={() => {
          if (!isDragging) {
            onOpenChange(false);
          }
        }}
      />

      {/* Modal Container */}
      <div className={`fixed inset-0 z-[101] pointer-events-none ${
        bottomSheet 
          ? "flex items-end justify-center sm:items-center" 
          : "flex items-center justify-center"
      }`}>
        {/* Modal with smooth animation */}
        <div
          ref={sheetRef}
          className={`bg-white shadow-xl pointer-events-auto flex flex-col min-h-0 ${
            bottomSheet
              ? `w-full rounded-t-2xl sm:rounded-b-lg sm:rounded-lg sm:max-w-2xl sm:mx-4 max-h-[70vh] sm:max-h-[90vh] ${
                  isDragging
                    ? ""
                    : isVisible
                    ? "opacity-100 translate-y-0 sm:scale-100 transition-all duration-300 ease-out"
                    : "opacity-0 translate-y-full sm:translate-y-4 sm:scale-95 transition-all duration-300 ease-out"
                }`
              : `rounded-lg max-w-2xl w-full mx-4 max-h-[90vh] ${
                  isVisible
                    ? "opacity-100 scale-100 translate-y-0 transition-all duration-300 ease-out"
                    : "opacity-0 scale-95 translate-y-4 transition-all duration-300 ease-out"
                }`
          }`}
          style={{
            transform: dragY > 0 ? `translateY(${dragY}px)` : undefined,
            transition: isDragging ? "none" : undefined,
          }}
        >
          {/* Drag Handle - Only for mobile bottom sheets */}
          {bottomSheet && showDragHandle && (
            <div className="flex justify-center pt-3 pb-2 flex-shrink-0 sm:hidden">
              <div className="w-12 h-1.5 bg-gray-300 rounded-full" />
            </div>
          )}
          
          {/* Back Button */}
          {showBackButton && onBack && (
            <button
              onClick={onBack}
              className="absolute top-4 left-4 p-1 rounded-full hover:bg-gray-100 transition-colors z-10"
              aria-label="Back"
            >
              <ArrowLeft className="w-5 h-5 text-gray-600" />
            </button>
          )}
          
          {/* Close Button */}
          {showCloseButton && (
            <button
              onClick={() => onOpenChange(false)}
              className="absolute top-4 right-4 p-1 rounded-full hover:bg-gray-100 transition-colors z-10"
              aria-label="Close"
            >
              <X className="w-5 h-5 text-gray-600" />
            </button>
          )}
          {children}
        </div>
      </div>
    </>
  );
}
