import { InputHTMLAttributes, ReactNode, forwardRef } from "react";
import { cn } from "@/lib/classNames";
import { AlertCircle } from "lucide-react";

interface CheckboxProps extends InputHTMLAttributes<HTMLInputElement> {
  label?: ReactNode;
  error?: string;
  touched?: boolean;
}

export const Checkbox = forwardRef<HTMLInputElement, CheckboxProps>(
  ({ label, error, touched, className = "", ...props }, ref) => {
    const showError = touched && error;

    return (
      <div className="w-full">
        <label className="flex items-start gap-3 cursor-pointer">
          <input
            ref={ref}
            type="checkbox"
            className={cn(
              "mt-1 w-5 h-5 rounded border-gray-300 text-primary-500",
              "focus:ring-2 focus:ring-primary-500 focus:ring-offset-0",
              "transition-colors cursor-pointer flex-shrink-0",
              showError && "border-error-500",
              className
            )}
            aria-invalid={showError ? true : undefined}
            aria-describedby={showError ? `${props.id || "checkbox"}-error` : undefined}
            {...props}
          />
          {label && (
            <span className={cn(
              "body-text-sm flex-1",
              showError && "text-error-500"
            )}>
              {label}
            </span>
          )}
        </label>
        {showError && (
          <p
            id={`${props.id || "checkbox"}-error`}
            className="caption text-error-500 mt-1 ml-8 flex items-center gap-1"
            role="alert"
          >
            <AlertCircle className="w-3 h-3" />
            {error}
          </p>
        )}
      </div>
    );
  }
);

Checkbox.displayName = "Checkbox";
