"use client";

import {
  Calendar,
  MapPin,
  Building,
  Search,
  ShoppingCart,
  Check,
  X,
  Clock,
  CalendarClock,
  RotateCcw,
  ChevronRight,
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { Dialog } from "@/components/ui/dialog";
import Image from "next/image";
import { useState, useMemo, useEffect } from "react";
import { PlaceholderImage } from "@/components/data-display/PlaceholderImage";
import { useAppointments } from "@/features/profile/hooks/useAppointments";
import { transformAppointmentToDisplay } from "@/features/profile/utils/transformAppointmentToDisplay";
import type { AppointmentDisplay } from "@/features/profile/utils/transformAppointmentToDisplay";
import { Spinner } from "@/components/feedback/Spinner";
import { useRouter } from "next/navigation";
import { routes } from "@/config/routes";
import type { Appointment } from "@/lib/schemas/appointment";
import type { GetAppointmentDetailResponse } from "@/lib/schemas/appointmentDetail";
import { API_BASE_URL } from "@/config/env";

interface AppointmentWithOriginal {
  display: AppointmentDisplay;
  original: Appointment;
}

export default function AppointmentsPage() {
  const router = useRouter();
  const { data, isLoading, error } = useAppointments();
  const [selectedAppointment, setSelectedAppointment] =
    useState<AppointmentWithOriginal | null>(null);
  const [imageErrors, setImageErrors] = useState<Record<string, boolean>>({});
  const [appointmentDetail, setAppointmentDetail] =
    useState<GetAppointmentDetailResponse | null>(null);
  const [loadingDetail, setLoadingDetail] = useState(false);
  const [showManageBookingModal, setShowManageBookingModal] = useState(false);
  const [showCancelConfirmationModal, setShowCancelConfirmationModal] =
    useState(false);
  const [cancelling, setCancelling] = useState(false);
  const [cancelError, setCancelError] = useState<string | null>(null);

  // Transform appointments and separate upcoming vs past
  const { upcomingAppointments, pastAppointments } = useMemo(() => {
    if (!data?.appointments) {
      return { upcomingAppointments: [], pastAppointments: [] };
    }

    const currencyUnit = data.currencyUnit || "Rs";
    const now = new Date();
    const upcoming: AppointmentWithOriginal[] = [];
    const past: AppointmentWithOriginal[] = [];

    // Transform and categorize appointments
    data.appointments.forEach((appointment) => {
      const transformed = transformAppointmentToDisplay(
        appointment,
        currencyUnit
      );

      const appointmentWithOriginal: AppointmentWithOriginal = {
        display: transformed,
        original: appointment,
      };

      // Check status first - "complete", "cancelled", or "no show" should be in past
      if (
        appointment.status === "complete" ||
        appointment.status === "cancelled" ||
        appointment.status === "no show" ||
        appointment.status === "noshow"
      ) {
        past.push(appointmentWithOriginal);
      } else {
        // For other statuses, check the date using original appointment data
        const appointmentDate = new Date(
          `${appointment.on}T${appointment.startTime}`
        );
        if (appointmentDate < now) {
          past.push(appointmentWithOriginal);
        } else {
          upcoming.push(appointmentWithOriginal);
        }
      }
    });

    // Set first appointment as selected if none selected
    if (!selectedAppointment && (upcoming.length > 0 || past.length > 0)) {
      setSelectedAppointment(upcoming[0] || past[0]);
    }

    return {
      upcomingAppointments: upcoming,
      pastAppointments: past.sort((a, b) => {
        // Sort past appointments by date (newest first)
        const dateA = new Date(`${a.original.on}T${a.original.startTime}`);
        const dateB = new Date(`${b.original.on}T${b.original.startTime}`);
        return dateB.getTime() - dateA.getTime();
      }),
    };
  }, [data, selectedAppointment]);

  // Fetch appointment detail when selected appointment changes
  useEffect(() => {
    const fetchAppointmentDetail = async () => {
      if (!selectedAppointment) {
        setAppointmentDetail(null);
        return;
      }

      try {
        setLoadingDetail(true);
        const { getAppointmentDetailClient } = await import(
          "@/features/profile/api/getAppointmentDetail-client"
        );
        const detail = await getAppointmentDetailClient(
          selectedAppointment.display.id
        );
        setAppointmentDetail(detail);
        
        // Check if policies are available
        if (detail.policies) {
          const hasPolicies = 
            (detail.policies.cancellationPolicies && detail.policies.cancellationPolicies.length > 0) ||
            (detail.policies.noShowPolicies && detail.policies.noShowPolicies.length > 0) ||
            (detail.policies.reschedulePolicies && detail.policies.reschedulePolicies.length > 0) ||
            (detail.policies.paymentPolicies && detail.policies.paymentPolicies.length > 0);
          
          console.log("Policies available:", hasPolicies);
          console.log("Policies data:", detail.policies);
        } else {
          console.log("No policies available for this appointment");
        }
      } catch (error) {
        console.error("Failed to fetch appointment detail:", error);
        setAppointmentDetail(null);
      } finally {
        setLoadingDetail(false);
      }
    };

    fetchAppointmentDetail();
  }, [selectedAppointment]);

  // Get status badge info
  const getStatusBadge = (status: string) => {
    const statusLower = status.toLowerCase();
    if (statusLower === "complete" || statusLower === "completed") {
      return {
        label: "Completed",
        className: "bg-green-600 text-white hover:bg-green-700",
        icon: <Check className="w-4 h-4" />,
      };
    }
    if (
      statusLower === "cancelled" ||
      statusLower === "canceled" ||
      statusLower === "cancel"
    ) {
      return {
        label: "Cancelled",
        className: "bg-red-600 text-white hover:bg-red-700 border-red-600",
        icon: <X className="w-4 h-4" />,
      };
    }
    if (statusLower === "no show" || statusLower === "noshow") {
      return {
        label: "No Show",
        className: "bg-orange-600 text-white hover:bg-orange-700",
        icon: <Clock className="w-4 h-4" />,
      };
    }
    // Default to confirmed
    return {
      label: "Confirmed",
      className: "bg-purple-600 text-white hover:bg-purple-700",
      icon: <Check className="w-4 h-4" />,
    };
  };

  // Check if appointment should show "Book again" button
  const shouldShowBookAgain = (status: string) => {
    const statusLower = status.toLowerCase();
    return (
      statusLower === "complete" ||
      statusLower === "completed" ||
      statusLower === "cancelled" ||
      statusLower === "canceled" ||
      statusLower === "cancel" ||
      statusLower === "no show" ||
      statusLower === "noshow"
    );
  };

  // Handle book again - navigate to salon detail page
  const handleBookAgain = async (appointment: AppointmentWithOriginal) => {
    try {
      // Fetch appointment detail to get salon ID
      const { getAppointmentDetailClient } = await import(
        "@/features/profile/api/getAppointmentDetail-client"
      );
      const detail = await getAppointmentDetailClient(appointment.display.id);

      // Get salon ID from the salonDetail object in the response
      const salonId = detail.appointment.salonDetail?.id;

      // Navigate to salon detail page if salon ID is available
      if (salonId) {
        router.push(routes.salonDetail(salonId));
      } else {
        console.error(
          "Salon ID not available for appointment:",
          appointment.display.id
        );
        // Fallback to salons page if salon ID not available
        router.push(routes.saloons);
      }
    } catch (error) {
      console.error("Failed to fetch appointment detail:", error);
      // Fallback to salons page on error
      router.push(routes.saloons);
    }
  };

  // Handle venue details click - navigate to salon detail page
  const handleVenueDetailsClick = async (
    appointment: AppointmentWithOriginal
  ) => {
    try {
      // Fetch appointment detail to get salon ID
      const { getAppointmentDetailClient } = await import(
        "@/features/profile/api/getAppointmentDetail-client"
      );
      const detail = await getAppointmentDetailClient(appointment.display.id);

      // Get salon ID from the salonDetail object in the response
      const salonId = detail.appointment.salonDetail?.id;

      // Navigate to salon detail page if salon ID is available
      if (salonId) {
        // Navigate to /saloons/{salonId} (e.g., /saloons/101)
        router.push(routes.salonDetail(salonId));
      } else {
        console.error(
          "Salon ID not available for appointment:",
          appointment.display.id,
          "Response:",
          detail
        );
        // Fallback to salons page if salon ID not available
        router.push(routes.saloons);
      }
    } catch (error) {
      console.error("Failed to fetch appointment detail:", error);
      // Fallback to salons page on error
      router.push(routes.saloons);
    }
  };

  // Handle cancel appointment
  const handleCancelAppointment = async () => {
    if (!selectedAppointment || !appointmentDetail) {
      setCancelError("Appointment details not available");
      return;
    }

    const cancellationCharges = parseFloat(
      appointmentDetail.appointment.cancellationCharges || "0"
    );

    try {
      setCancelling(true);
      setCancelError(null);

      const { cancelAppointmentClient } = await import(
        "@/features/profile/api/cancelAppointment-client"
      );

      console.log("Cancelling appointment:", {
        appointmentId: selectedAppointment.display.id,
        cancellationCharges,
      });

      await cancelAppointmentClient(selectedAppointment.display.id, {
        cancellationCharges,
      });

      // Close modals
      setShowCancelConfirmationModal(false);
      setShowManageBookingModal(false);

      // Refresh appointments list by refetching
      // The useAppointments hook should handle this, but we can trigger a refetch
      window.location.reload(); // Simple approach - could use React Query refetch instead
    } catch (error: any) {
      console.error("Failed to cancel appointment:", error);
      console.error("Error details:", {
        message: error?.message,
        status: error?.status,
        statusCode: error?.statusCode,
        error: error?.error,
      });
      
      // Extract a more user-friendly error message
      let errorMessage = "Failed to cancel appointment. Please try again.";
      if (error?.message) {
        errorMessage = error.message;
      } else if (error?.error) {
        errorMessage = error.error;
      }
      
      setCancelError(errorMessage);
    } finally {
      setCancelling(false);
    }
  };

  // Check if appointment can be cancelled (not already cancelled, completed, or no show)
  const canCancelAppointment = (appointment: AppointmentWithOriginal) => {
    // Use status from appointment detail API if available, otherwise use display status
    const status = (
      appointmentDetail?.appointment.status ||
      appointment.display.status
    ).toLowerCase();
    return (
      status !== "cancelled" &&
      status !== "canceled" &&
      status !== "cancel" &&
      status !== "complete" &&
      status !== "completed" &&
      status !== "no show" &&
      status !== "noshow"
    );
  };

  if (isLoading) {
    return (
      <div>
        <h1 className="heading-4 text-gray-900 mb-10">Appointments</h1>
        <div className="flex items-center justify-center py-20">
          <Spinner />
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div>
        <h1 className="heading-4 text-gray-900 mb-10">Appointments</h1>
        <div className="rounded-xl border border-red-200 bg-red-50 p-8 text-center">
          <p className="body-text text-red-600">
            Failed to load appointments. Please try again later.
          </p>
        </div>
      </div>
    );
  }

  return (
    <div>
      <h1 className="heading-4 text-gray-900 mb-10">Appointments</h1>
      {/* Grid layout: For 1440px viewport, proportions are approximately:
          Left column: ~35% (appointments list)
          Right column: ~65% (appointment details)
          Gap: 56px (3.5rem)
      */}
      <div className="grid grid-cols-1 lg:grid-cols-[0.45fr_0.55fr] gap-14">
        {/* Left Section - Appointments List */}

        <div className="space-y-6">
          {/* Upcoming Section */}
          <div>
            <h2 className="heading-6 text-gray-900 mb-4">
              Upcoming ({upcomingAppointments.length})
            </h2>
            {upcomingAppointments.length > 0 ? (
              <div className="space-y-4 max-h-[400px] overflow-y-auto">
                {upcomingAppointments.map((appointment) => (
                  <button
                    key={appointment.display.id}
                    onClick={() => setSelectedAppointment(appointment)}
                    className={`w-full text-left bg-white rounded-xl border p-4 hover:border-purple-300 transition-colors ${
                      selectedAppointment?.display.id === appointment.display.id
                        ? "border-purple-500"
                        : "border-gray-200"
                    }`}
                  >
                    <div className="flex gap-4">
                      <div className="w-20 h-20 rounded-lg overflow-hidden flex-shrink-0 relative">
                        {imageErrors[`list-${appointment.display.id}`] ||
                        !appointment.display.image ||
                        appointment.display.image.includes("placeholder") ? (
                          <PlaceholderImage className="w-full h-full" text="" />
                        ) : (
                          <Image
                            src={appointment.display.image}
                            alt={appointment.display.venueName}
                            fill
                            className="object-cover"
                            onError={() =>
                              setImageErrors((prev) => ({
                                ...prev,
                                [`list-${appointment.display.id}`]: true,
                              }))
                            }
                          />
                        )}
                      </div>
                      <div className="flex-1 min-w-0">
                        <h3 className="heading-6 text-gray-900 mb-1">
                          {appointment.display.venueName}
                        </h3>
                        <p className="body-text-sm text-gray-600 mb-1">
                          {appointment.display.date} at{" "}
                          {appointment.display.time}
                        </p>
                        <p className="body-text-sm text-gray-900 font-medium">
                          {appointment.display.cost} •{" "}
                          {appointment.display.items} item
                          {appointment.display.items > 1 ? "s" : ""}
                        </p>
                      </div>
                    </div>
                  </button>
                ))}
              </div>
            ) : (
              <div className="rounded-xl border border-gray-200 shadow-sm p-8 text-center">
                <Calendar className="w-16 h-16 text-purple-300 mx-auto mb-4" />
                <h3 className="heading-6 text-gray-900 mb-2">
                  No upcoming appointments
                </h3>
                <p className="body-text-sm text-gray-600 mb-6">
                  Your upcoming appointments will appear here when you book.
                </p>
                <div className="flex justify-center">
                  <Button
                    variant="ghost"
                    className="bg-transparent hover:bg-gray-50 border"
                  >
                    <Search className="w-4 h-4 mr-2" />
                    Search salons
                  </Button>
                </div>
              </div>
            )}
          </div>

          {/* Past Section */}
          <div>
            <h2 className="heading-6 text-gray-900 mb-4">
              Past ({pastAppointments.length})
            </h2>
            {pastAppointments.length > 0 ? (
              <div className="space-y-4 max-h-[600px] overflow-y-auto">
                {pastAppointments.map((appointment) => (
                  <button
                    key={appointment.display.id}
                    onClick={() => setSelectedAppointment(appointment)}
                    className={`w-full text-left bg-white rounded-xl border p-4 hover:border-purple-300 transition-colors ${
                      selectedAppointment?.display.id === appointment.display.id
                        ? "border-purple-500"
                        : "border-gray-200"
                    }`}
                  >
                    <div className="flex gap-4">
                      <div className="w-20 h-20 rounded-lg overflow-hidden flex-shrink-0 relative">
                        {imageErrors[`list-${appointment.display.id}`] ||
                        !appointment.display.image ||
                        appointment.display.image.includes("placeholder") ? (
                          <PlaceholderImage className="w-full h-full" text="" />
                        ) : (
                          <Image
                            src={appointment.display.image}
                            alt={appointment.display.venueName}
                            fill
                            className="object-cover"
                            onError={() =>
                              setImageErrors((prev) => ({
                                ...prev,
                                [`list-${appointment.display.id}`]: true,
                              }))
                            }
                          />
                        )}
                      </div>
                      <div className="flex-1 min-w-0">
                        <h3 className="heading-6 text-gray-900 mb-1">
                          {appointment.display.venueName}
                        </h3>
                        <p className="body-text-sm text-gray-600 mb-1">
                          {appointment.display.date} at{" "}
                          {appointment.display.time}
                        </p>
                        <p className="body-text-sm text-gray-900 font-medium">
                          {appointment.display.cost} •{" "}
                          {appointment.display.items} item
                          {appointment.display.items > 1 ? "s" : ""}
                        </p>
                      </div>
                    </div>
                  </button>
                ))}
              </div>
            ) : (
              <div className="rounded-xl border border-gray-200 shadow-sm p-8 text-center">
                <p className="body-text-sm text-gray-600">
                  No past appointments yet.
                </p>
              </div>
            )}
          </div>
        </div>

        {/* Right Section - Appointment Details */}
        <div className="space-y-6 bg-white border rounded-xl">
          {selectedAppointment ? (
            <>
              {/* Banner with Salon Background */}
              <div className="bg-white rounded-t-xl border border-gray-200 overflow-hidden">
                <div className="relative h-72">
                  {loadingDetail ? (
                    <>
                      <div className="w-full h-full bg-gray-200 animate-pulse"></div>
                      <div className="absolute inset-0 bg-black/70"></div>
                      <div className="absolute inset-0 flex items-end p-6">
                        <div className="h-8 w-48 bg-gray-300 animate-pulse rounded"></div>
                      </div>
                    </>
                  ) : (() => {
                    // Use image and name from appointment detail API if available, otherwise use from list
                    const salonImage = appointmentDetail?.appointment.salonDetail?.coverImage
                      ? (() => {
                          const img = appointmentDetail.appointment.salonDetail.coverImage;
                          if (img?.startsWith("http://") || img?.startsWith("https://")) {
                            return img;
                          } else if (img?.startsWith("/")) {
                            return `${API_BASE_URL}${img}`;
                          } else if (img) {
                            return `${API_BASE_URL}/${img}`;
                          }
                          return null;
                        })()
                      : selectedAppointment.display.image;
                    
                    const salonName = appointmentDetail?.appointment.salonDetail?.salonName || selectedAppointment.display.venueName;
                    
                    return imageErrors[`detail-${selectedAppointment.display.id}`] ||
                      !salonImage ||
                      salonImage.includes("placeholder") ? (
                      <>
                        <PlaceholderImage
                          className="w-full h-full"
                          style={{ height: "100%" }}
                        />
                        <div className="absolute inset-0 bg-black/70"></div>
                        <div className="absolute inset-0 flex items-end p-6">
                          <h2 className="heading-3 text-white relative z-10">
                            {salonName}
                          </h2>
                        </div>
                      </>
                    ) : (
                      <>
                        <div className="absolute inset-0 bg-black/70"></div>
                        <Image
                          src={salonImage}
                          alt={salonName}
                          fill
                          className="object-cover"
                          onError={() =>
                            setImageErrors((prev) => ({
                              ...prev,
                              [`detail-${selectedAppointment.display.id}`]: true,
                            }))
                          }
                        />
                        <div className="absolute inset-0 flex items-end p-6">
                          <h2 className="heading-3 text-white relative z-10">
                            {salonName}
                          </h2>
                        </div>
                      </>
                    );
                  })()}
                </div>
              </div>

              {/* Appointment Status */}
              <div className="px-6 space-y-6 pb-4 rounded-b-xl">
                <div className="flex items-center gap-3">
                  {loadingDetail ? (
                    <div className="h-8 w-32 bg-gray-200 animate-pulse rounded-full"></div>
                  ) : (
                    (() => {
                      // Use status from appointment detail API if available, otherwise use original status
                      const status = appointmentDetail?.appointment.status || selectedAppointment.original.status;
                      const statusBadge = getStatusBadge(status);
                      return (
                        <Button className={statusBadge.className}>
                          <span className="w-4 h-4 mr-2">{statusBadge.icon}</span>
                          {statusBadge.label}
                        </Button>
                      );
                    })()
                  )}
                </div>

                {/* Date and Time */}
                {loadingDetail ? (
                  <div className="space-y-2">
                    <div className="h-6 w-64 bg-gray-200 animate-pulse rounded"></div>
                    <div className="h-4 w-48 bg-gray-200 animate-pulse rounded"></div>
                  </div>
                ) : (
                  <div>
                    <h3 className="heading-5 text-gray-900 mb-1">
                      {(() => {
                        // Use date/time from appointment detail API if available
                        if (appointmentDetail?.appointment.on && appointmentDetail?.appointment.startTime) {
                          const appointmentDate = new Date(`${appointmentDetail.appointment.on}T${appointmentDetail.appointment.startTime}`);
                          const formattedDate = appointmentDate.toLocaleDateString("en-US", {
                            weekday: "short",
                            day: "numeric",
                            month: "short",
                            year: "numeric",
                          });
                          const timeParts = appointmentDetail.appointment.startTime.split(":");
                          const hours = parseInt(timeParts[0], 10);
                          const minutes = timeParts[1];
                          const ampm = hours >= 12 ? "pm" : "am";
                          const displayHours = hours % 12 || 12;
                          const formattedTime = `${displayHours}:${minutes} ${ampm}`;
                          return `${formattedDate} at ${formattedTime}`;
                        }
                        return selectedAppointment.display.fullDate;
                      })()}
                    </h3>
                    <p className="body-text-sm text-gray-600">
                      {(() => {
                        // Use duration from appointment detail API if available
                        const duration = appointmentDetail?.appointment.duration || selectedAppointment.display.duration;
                        if (duration) {
                          const hours = Math.floor(duration / 60);
                          const minutes = duration % 60;
                          if (hours > 0 && minutes > 0) {
                            return `${hours} hour${hours > 1 ? "s" : ""}, ${minutes} minute${minutes > 1 ? "s" : ""} duration`;
                          } else if (hours > 0) {
                            return `${hours} hour${hours > 1 ? "s" : ""} duration`;
                          } else {
                            return `${minutes} minute${minutes > 1 ? "s" : ""} duration`;
                          }
                        }
                        return "Duration not available";
                      })()}
                    </p>
                  </div>
                )}

                {/* Book again - Show for completed, cancelled, no show */}
                {!loadingDetail && shouldShowBookAgain(
                  appointmentDetail?.appointment.status ||
                    selectedAppointment.original.status
                ) && (
                  <div className="border-t border-gray-200 pt-4">
                    <button
                      onClick={() => handleBookAgain(selectedAppointment)}
                      className="w-full flex items-center gap-3 py-1 rounded-lg hover:bg-gray-50 transition-colors text-left"
                    >
                      <div className="w-10 h-10 rounded-full bg-purple-100 flex items-center justify-center flex-shrink-0">
                        <ShoppingCart className="w-5 h-5 text-purple-600" />
                      </div>
                      <div className="flex-1">
                        <h4 className="body-text-sm font-medium text-gray-900 mb-1">
                          Book again
                        </h4>
                        <p className="body-text-xs text-gray-600">
                          Book your next appointment
                        </p>
                      </div>
                    </button>
                  </div>
                )}

                {/* Getting there */}
                <div className="space-y-2">
                  {loadingDetail ? (
                    <div className="w-full flex items-start gap-3 py-1 rounded-lg">
                      <div className="w-10 h-10 rounded-full bg-gray-200 animate-pulse flex-shrink-0 mt-0.5"></div>
                      <div className="flex-1 space-y-2">
                        <div className="h-4 w-24 bg-gray-200 animate-pulse rounded"></div>
                        <div className="h-4 w-48 bg-gray-200 animate-pulse rounded"></div>
                      </div>
                    </div>
                  ) : (
                    <div className="w-full flex items-start gap-3 py-1 rounded-lg text-left">
                      <div className="w-10 h-10 rounded-full bg-purple-100 flex items-center justify-center flex-shrink-0 mt-0.5">
                        <MapPin className="w-5 h-5 text-purple-600" />
                      </div>
                      <div className="flex-1">
                        <h4 className="body-text-sm font-medium text-gray-900 mb-1">
                          Getting there
                        </h4>
                        <p className="body-text-sm text-gray-600">
                          {(() => {
                            // Use address from appointment detail API if available
                            if (appointmentDetail?.appointment.salonDetail?.addressDB) {
                              const addr = appointmentDetail.appointment.salonDetail.addressDB;
                              const addressParts = [
                                addr.streetAddress,
                                addr.city,
                                addr.district,
                                addr.province,
                              ].filter(Boolean);
                              return addressParts.length > 0 ? addressParts.join(", ") : "Address not available";
                            }
                            return selectedAppointment.display.address || "Address not available";
                          })()}
                        </p>
                      </div>
                    </div>
                  )}

                  {/* Manage booking - Only show for upcoming appointments that can be cancelled */}
                  {!loadingDetail && canCancelAppointment(selectedAppointment) && (
                    <button
                      type="button"
                      onClick={() => setShowManageBookingModal(true)}
                      className="w-full flex items-start gap-3 py-1 rounded-lg hover:bg-gray-50 transition-colors text-left cursor-pointer"
                    >
                      <div className="w-10 h-10 rounded-full bg-purple-100 flex items-center justify-center flex-shrink-0 mt-0.5">
                        <CalendarClock className="w-5 h-5 text-purple-600" />
                      </div>
                      <div className="flex-1 flex items-center justify-between">
                        <div>
                          <h4 className="body-text-sm font-medium text-gray-900 mb-1">
                            Manage booking
                          </h4>
                          <p className="body-text-sm text-gray-600">
                            Reschedule or cancel your booking
                          </p>
                        </div>
                        <ChevronRight className="w-5 h-5 text-gray-400 flex-shrink-0" />
                      </div>
                    </button>
                  )}
                </div>

                {/* Venue details - Clickable */}
                <div className="space-y-2">
                  {loadingDetail ? (
                    <div className="w-full flex items-start gap-3 py-1 rounded-lg">
                      <div className="w-10 h-10 rounded-full bg-gray-200 animate-pulse flex-shrink-0 mt-0.5"></div>
                      <div className="flex-1 space-y-2">
                        <div className="h-4 w-24 bg-gray-200 animate-pulse rounded"></div>
                        <div className="h-4 w-32 bg-gray-200 animate-pulse rounded"></div>
                      </div>
                    </div>
                  ) : (
                    <button
                      type="button"
                      onClick={() => handleVenueDetailsClick(selectedAppointment)}
                      className="w-full flex items-start gap-3 py-1 rounded-lg hover:bg-gray-50 transition-colors text-left cursor-pointer"
                    >
                      <div className="w-10 h-10 rounded-full bg-purple-100 flex items-center justify-center flex-shrink-0 mt-0.5">
                        <Building className="w-5 h-5 text-purple-600" />
                      </div>
                      <div className="flex-1">
                        <h4 className="body-text-sm font-medium text-gray-900 mb-1">
                          Venue details
                        </h4>
                        <p className="body-text-sm text-gray-600">
                          {appointmentDetail?.appointment.salonDetail?.salonName || selectedAppointment.display.venueName}
                        </p>
                      </div>
                    </button>
                  )}
                </div>

                {/* Policies */}
                {loadingDetail ? (
                  <div className="border-t border-gray-200 pt-4">
                    <div className="h-6 w-24 bg-gray-200 animate-pulse rounded mb-3"></div>
                    <div className="space-y-2">
                      <div className="h-20 bg-gray-100 animate-pulse rounded-lg"></div>
                    </div>
                  </div>
                ) : appointmentDetail?.policies ? (
                  <div className="border-t border-gray-200 pt-4 space-y-4">
                    <h4 className="body-text-sm font-medium text-gray-900 mb-3">
                      Policies
                    </h4>
                    <div className="space-y-4">
                      {/* Cancellation Policies */}
                      {appointmentDetail.policies.cancellationPolicies?.filter(
                        (p: any) => p.status
                      ).length > 0 && (
                        <div className="space-y-2">
                          <h5 className="body-text-xs font-semibold text-gray-900">
                            Cancellation Policy
                          </h5>
                          <div className="space-y-2">
                            {appointmentDetail.policies.cancellationPolicies
                              .filter((p: any) => p.status)
                              .map((policy: any, index: number) => (
                                <div
                                  key={policy.id || index}
                                  className="p-3 bg-gray-50 rounded-lg border border-gray-200"
                                >
                                  <div className="space-y-1 text-xs text-gray-700">
                                    {policy.hoursBeforeBooking && (
                                      <p>
                                        <span className="font-medium">
                                          Hours before booking:
                                        </span>{" "}
                                        {policy.hoursBeforeBooking}
                                      </p>
                                    )}
                                    {policy.refundPercentage !== null &&
                                      policy.refundPercentage !== undefined && (
                                        <p>
                                          <span className="font-medium">
                                            Refund:
                                          </span>{" "}
                                          {policy.refundPercentage}%
                                        </p>
                                      )}
                                  </div>
                                </div>
                              ))}
                          </div>
                        </div>
                      )}

                      {/* No Show Policies */}
                      {appointmentDetail.policies.noShowPolicies?.filter(
                        (p: any) => p.status
                      ).length > 0 && (
                        <div className="space-y-2">
                          <h5 className="body-text-xs font-semibold text-gray-900">
                            No Show Policy
                          </h5>
                          <div className="space-y-2">
                            {appointmentDetail.policies.noShowPolicies
                              .filter((p: any) => p.status)
                              .map((policy: any, index: number) => (
                                <div
                                  key={policy.id || index}
                                  className="p-3 bg-gray-50 rounded-lg border border-gray-200"
                                >
                                  <div className="space-y-1 text-xs text-gray-700">
                                    {policy.refundPercentage !== null &&
                                      policy.refundPercentage !== undefined && (
                                        <p>
                                          <span className="font-medium">
                                            Refund:
                                          </span>{" "}
                                          {policy.refundPercentage}%
                                        </p>
                                      )}
                                  </div>
                                </div>
                              ))}
                          </div>
                        </div>
                      )}

                      {/* Reschedule Policies */}
                      {appointmentDetail.policies.reschedulePolicies?.filter(
                        (p: any) => p.status
                      ).length > 0 && (
                        <div className="space-y-2">
                          <h5 className="body-text-xs font-semibold text-gray-900">
                            Reschedule Policy
                          </h5>
                          <div className="space-y-2">
                            {appointmentDetail.policies.reschedulePolicies
                              .filter((p: any) => p.status)
                              .map((policy: any, index: number) => (
                                <div
                                  key={policy.id || index}
                                  className="p-3 bg-gray-50 rounded-lg border border-gray-200"
                                >
                                  <div className="space-y-1 text-xs text-gray-700">
                                    {policy.hoursBeforeBooking !== null &&
                                      policy.hoursBeforeBooking !==
                                        undefined && (
                                        <p>
                                          <span className="font-medium">
                                            Hours before booking:
                                          </span>{" "}
                                          {policy.hoursBeforeBooking}
                                        </p>
                                      )}
                                    {policy.count !== null &&
                                      policy.count !== undefined && (
                                        <p>
                                          <span className="font-medium">
                                            Allowed reschedules:
                                          </span>{" "}
                                          {policy.count}
                                        </p>
                                      )}
                                  </div>
                                </div>
                              ))}
                          </div>
                        </div>
                      )}

                      {/* Payment Policies */}
                      {appointmentDetail.policies.paymentPolicies?.filter(
                        (p: any) => p.status
                      ).length > 0 && (
                        <div className="space-y-2">
                          <h5 className="body-text-xs font-semibold text-gray-900">
                            Payment Policy
                          </h5>
                          <div className="space-y-2">
                            {appointmentDetail.policies.paymentPolicies
                              .filter((p: any) => p.status)
                              .map((policy: any, index: number) => (
                                <div
                                  key={policy.id || index}
                                  className="p-3 bg-gray-50 rounded-lg border border-gray-200"
                                >
                                  <div className="space-y-1 text-xs text-gray-700">
                                    {policy.percentage !== null &&
                                      policy.percentage !== undefined && (
                                        <p>
                                          <span className="font-medium">
                                            Advance payment:
                                          </span>{" "}
                                          {policy.percentage}%
                                        </p>
                                      )}
                                  </div>
                                </div>
                              ))}
                          </div>
                        </div>
                      )}
                    </div>
                  </div>
                ) : (
                  <div className="border-t border-gray-200 pt-4">
                    <h4 className="body-text-sm font-medium text-gray-900 mb-3">
                      Policies
                    </h4>
                    <div className="p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                      <p className="body-text-xs text-yellow-800">
                        No policies available for this appointment.
                      </p>
                    </div>
                  </div>
                )}

                {/* Overview */}
                <div className="border-t border-gray-200 pt-4">
                  <h4 className="body-text-sm font-medium text-gray-900 mb-3">
                    Overview
                  </h4>
                  {loadingDetail ? (
                    <div className="space-y-3">
                      <div className="h-16 bg-gray-100 animate-pulse rounded"></div>
                      <div className="h-16 bg-gray-100 animate-pulse rounded"></div>
                    </div>
                  ) : (
                    <div className="space-y-3">
                      {/* Services */}
                      {appointmentDetail?.appointment.jobs &&
                      appointmentDetail.appointment.jobs.length > 0 ? (
                      <div className="space-y-2">
                        {appointmentDetail.appointment.jobs.map(
                          (job: any, index: number) => (
                            <div
                              key={job.id || index}
                              className="flex justify-between items-center"
                            >
                              <div className="flex-1">
                                <p className="body-text-sm text-gray-900">
                                  {job.service?.serviceName || "Service"}
                                </p>
                                {job.employee && (
                                  <p className="body-text-xs text-gray-500">
                                    {job.employee}
                                  </p>
                                )}
                              </div>
                              <p className="body-text-sm font-medium text-gray-900">
                                {appointmentDetail.currencyUnit}
                                {parseFloat(job.total || "0").toFixed(2)}
                              </p>
                            </div>
                          )
                        )}
                      </div>
                    ) : selectedAppointment.display.items > 0 ? (
                      <div className="flex justify-between items-center">
                        <div>
                          <p className="body-text-sm text-gray-900">
                            {selectedAppointment.display.items} service
                            {selectedAppointment.display.items > 1 ? "s" : ""}
                          </p>
                        </div>
                        <p className="body-text-sm font-medium text-gray-900">
                          {selectedAppointment.display.cost}
                        </p>
                      </div>
                    ) : (
                      <p className="body-text-sm text-gray-600">
                        Service details not available
                      </p>
                    )}

                    {/* Subtotal */}
                    {appointmentDetail?.appointment.subTotal && (
                      <div className="flex justify-between items-center pt-2 border-t border-gray-100">
                        <p className="body-text-sm text-gray-600">Subtotal</p>
                        <p className="body-text-sm font-medium text-gray-900">
                          {appointmentDetail.currencyUnit}
                          {parseFloat(
                            appointmentDetail.appointment.subTotal
                          ).toFixed(2)}
                        </p>
                      </div>
                    )}

                    {/* Discount */}
                    {appointmentDetail?.appointment.discount &&
                      parseFloat(appointmentDetail.appointment.discount) >
                        0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">Discount</p>
                          <p className="body-text-sm font-medium text-green-600">
                            -{appointmentDetail.currencyUnit}
                            {parseFloat(
                              appointmentDetail.appointment.discount
                            ).toFixed(2)}
                          </p>
                        </div>
                      )}

                    {/* Platform Fee */}
                    {appointmentDetail?.appointment.platformFee &&
                      parseFloat(appointmentDetail.appointment.platformFee) >
                        0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">
                            Platform fee
                          </p>
                          <p className="body-text-sm font-medium text-gray-900">
                            {appointmentDetail.currencyUnit}
                            {parseFloat(
                              appointmentDetail.appointment.platformFee
                            ).toFixed(2)}
                          </p>
                        </div>
                      )}

                    {/* Tip */}
                    {appointmentDetail?.appointment.tip &&
                      parseFloat(appointmentDetail.appointment.tip) > 0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">Tip</p>
                          <p className="body-text-sm font-medium text-gray-900">
                            {appointmentDetail.currencyUnit}
                            {parseFloat(
                              appointmentDetail.appointment.tip
                            ).toFixed(2)}
                          </p>
                        </div>
                      )}

                    {/* VAT */}
                    {appointmentDetail?.appointment.vatAmount &&
                      parseFloat(appointmentDetail.appointment.vatAmount) >
                        0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">VAT</p>
                          <p className="body-text-sm font-medium text-gray-900">
                            {appointmentDetail.currencyUnit}
                            {parseFloat(
                              appointmentDetail.appointment.vatAmount
                            ).toFixed(2)}
                          </p>
                        </div>
                      )}
                    </div>
                  )}
                </div>

                {/* Payment Breakdown */}
                {!loadingDetail && appointmentDetail?.appointment && (
                  <div className="border-t border-gray-200 pt-4 space-y-3">
                    {/* Paid Upfront */}
                    {appointmentDetail.appointment.paidUpfront &&
                      parseFloat(appointmentDetail.appointment.paidUpfront) >
                        0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">
                            Paid upfront
                          </p>
                          <p className="body-text-sm font-medium text-gray-900">
                            {appointmentDetail.currencyUnit}
                            {parseFloat(
                              appointmentDetail.appointment.paidUpfront
                            ).toFixed(2)}
                          </p>
                        </div>
                      )}

                    {/* Initial Payment */}
                    {appointmentDetail.appointment.initialPayment &&
                      parseFloat(appointmentDetail.appointment.initialPayment) >
                        0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">
                            Initial payment
                          </p>
                          <p className="body-text-sm font-medium text-gray-900">
                            {appointmentDetail.currencyUnit}
                            {parseFloat(
                              appointmentDetail.appointment.initialPayment
                            ).toFixed(2)}
                          </p>
                        </div>
                      )}

                    {/* Amount to Pay at Venue */}
                    {appointmentDetail.appointment.amountToPayAtVenue &&
                      parseFloat(
                        appointmentDetail.appointment.amountToPayAtVenue
                      ) > 0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">
                            Pay at venue
                          </p>
                          <p className="body-text-sm font-medium text-gray-900">
                            {appointmentDetail.currencyUnit}
                            {parseFloat(
                              appointmentDetail.appointment.amountToPayAtVenue
                            ).toFixed(2)}
                          </p>
                        </div>
                      )}

                    {/* Actual Captured Amount */}
                    {appointmentDetail.appointment.actualCapturedAmount &&
                      parseFloat(
                        appointmentDetail.appointment.actualCapturedAmount
                      ) > 0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">
                            Captured amount
                          </p>
                          <p className="body-text-sm font-medium text-gray-900">
                            {appointmentDetail.currencyUnit}
                            {parseFloat(
                              appointmentDetail.appointment.actualCapturedAmount
                            ).toFixed(2)}
                          </p>
                        </div>
                      )}

                    {/* Upfront Payment Percentage */}
                    {appointmentDetail.appointment.upFrontPaymentPercentage !==
                      null &&
                      appointmentDetail.appointment.upFrontPaymentPercentage !==
                        undefined &&
                      appointmentDetail.appointment.upFrontPaymentPercentage >
                        0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">
                            Upfront payment percentage
                          </p>
                          <p className="body-text-sm font-medium text-gray-900">
                            {
                              appointmentDetail.appointment
                                .upFrontPaymentPercentage
                            }
                            %
                          </p>
                        </div>
                      )}

                    {/* Final Payment */}
                    {appointmentDetail.appointment.finalPayment && (
                      <div className="flex justify-between items-center">
                        <p className="body-text-sm text-gray-600">
                          Final payment
                        </p>
                        <p className="body-text-sm font-medium text-gray-900">
                          {appointmentDetail.appointment.finalPayment}
                        </p>
                      </div>
                    )}

                    {/* Payment Method */}
                    {appointmentDetail.appointment.paymentMethod && (
                      <div className="flex justify-between items-center">
                        <p className="body-text-sm text-gray-600">
                          Payment method
                        </p>
                        <p className="body-text-sm font-medium text-gray-900 capitalize">
                          {appointmentDetail.appointment.paymentMethod}
                        </p>
                      </div>
                    )}
                  </div>
                )}

                {/* Charges and Adjustments */}
                {!loadingDetail && appointmentDetail?.appointment && (
                  <div className="border-t border-gray-200 pt-4 space-y-3">
                    <h5 className="body-text-sm font-semibold text-gray-900 mb-2">
                      Charges & Adjustments
                    </h5>

                    {/* No Show Charges */}
                    {appointmentDetail.appointment.noshowCharges &&
                      parseFloat(appointmentDetail.appointment.noshowCharges) >
                        0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">
                            No show charges
                          </p>
                          <p className="body-text-sm font-medium text-red-600">
                            {appointmentDetail.currencyUnit}
                            {parseFloat(
                              appointmentDetail.appointment.noshowCharges
                            ).toFixed(2)}
                          </p>
                        </div>
                      )}

                    {/* Cancellation Charges */}
                    {appointmentDetail.appointment.cancellationCharges &&
                      parseFloat(
                        appointmentDetail.appointment.cancellationCharges
                      ) > 0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">
                            Cancellation charges
                          </p>
                          <p className="body-text-sm font-medium text-red-600">
                            {appointmentDetail.currencyUnit}
                            {parseFloat(
                              appointmentDetail.appointment.cancellationCharges
                            ).toFixed(2)}
                          </p>
                        </div>
                      )}

                    {/* Penalty Amount */}
                    {appointmentDetail.appointment.penaltyAmount &&
                      parseFloat(appointmentDetail.appointment.penaltyAmount) >
                        0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">
                            Penalty amount
                          </p>
                          <p className="body-text-sm font-medium text-red-600">
                            {appointmentDetail.currencyUnit}
                            {parseFloat(
                              appointmentDetail.appointment.penaltyAmount
                            ).toFixed(2)}
                          </p>
                        </div>
                      )}

                    {/* Refund Amount */}
                    {appointmentDetail.appointment.refundAmount &&
                      parseFloat(appointmentDetail.appointment.refundAmount) >
                        0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">
                            Refund amount
                          </p>
                          <p className="body-text-sm font-medium text-green-600">
                            {appointmentDetail.currencyUnit}
                            {parseFloat(
                              appointmentDetail.appointment.refundAmount
                            ).toFixed(2)}
                          </p>
                        </div>
                      )}

                    {/* VAT Original Amount */}
                    {appointmentDetail.appointment.vatOriginalAmount &&
                      parseFloat(
                        appointmentDetail.appointment.vatOriginalAmount
                      ) > 0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">
                            VAT (original)
                          </p>
                          <p className="body-text-sm font-medium text-gray-900">
                            {appointmentDetail.currencyUnit}
                            {parseFloat(
                              appointmentDetail.appointment.vatOriginalAmount
                            ).toFixed(2)}
                          </p>
                        </div>
                      )}

                    {/* Reschedule Count */}
                    {appointmentDetail.appointment.reScheduleCount !==
                      undefined &&
                      appointmentDetail.appointment.reScheduleCount > 0 && (
                        <div className="flex justify-between items-center">
                          <p className="body-text-sm text-gray-600">
                            Reschedule count
                          </p>
                          <p className="body-text-sm font-medium text-gray-900">
                            {appointmentDetail.appointment.reScheduleCount}
                          </p>
                        </div>
                      )}
                  </div>
                )}

                {/* Total */}
                {loadingDetail ? (
                  <div className="border-t border-gray-200 pt-4 flex justify-between items-center">
                    <div className="h-5 w-16 bg-gray-200 animate-pulse rounded"></div>
                    <div className="h-5 w-24 bg-gray-200 animate-pulse rounded"></div>
                  </div>
                ) : (
                  <div className="border-t border-gray-200 pt-4 flex justify-between items-center">
                    <h4 className="body-text-sm font-bold text-gray-900">
                      Total
                    </h4>
                    <p className="body-text-sm font-bold text-gray-900">
                      {appointmentDetail?.appointment.total
                        ? `${appointmentDetail.currencyUnit}${parseFloat(
                            appointmentDetail.appointment.total
                          ).toFixed(2)}`
                        : selectedAppointment.display.cost}
                    </p>
                  </div>
                )}

                {/* Booking Reference */}
                {loadingDetail ? (
                  <div className="pt-4">
                    <div className="h-4 w-40 bg-gray-200 animate-pulse rounded"></div>
                  </div>
                ) : (
                  <div className="pt-4">
                    <p className="body-text-xs text-gray-500">
                      Booking ref:{" "}
                      {appointmentDetail?.appointment.bookingReference ||
                        appointmentDetail?.appointment.id ||
                        selectedAppointment.display.bookingReference ||
                        selectedAppointment.display.id}
                    </p>
                  </div>
                )}
              </div>
            </>
          ) : (
            <div className="bg-white rounded-xl border border-gray-200 p-8 text-center">
              <p className="body-text text-gray-600">
                Select an appointment to view details
              </p>
            </div>
          )}
        </div>
      </div>

      {/* Manage Booking Modal */}
      <Dialog
        open={showManageBookingModal}
        onOpenChange={setShowManageBookingModal}
        showCloseButton={true}
        bottomSheet={true}
      >
        <div className="flex flex-col h-full max-w-2xl w-full min-h-0">
          {/* Modal Header */}
          <div className="flex items-center justify-between p-6 pb-4 flex-shrink-0 border-b border-gray-200">
            <h2 className="text-2xl font-bold text-gray-900">Manage booking</h2>
            <button
              onClick={() => setShowManageBookingModal(false)}
              className="p-1 rounded-full hover:bg-gray-100 transition-colors"
            >
              <X className="w-5 h-5 text-gray-600" />
            </button>
          </div>

          {/* Modal Content */}
          <div className="flex-1 overflow-y-auto px-6 py-4 min-h-0">
            {selectedAppointment && appointmentDetail && (
              <div className="space-y-4">
                {/* Booking Info */}
                <div className="p-4 bg-gray-50 rounded-lg border border-gray-200">
                  <div className="space-y-2">
                    <p className="body-text-sm font-medium text-gray-900">
                      {(() => {
                        const date = new Date(
                          `${selectedAppointment.original.on}T${selectedAppointment.original.startTime}`
                        );
                        return date.toLocaleDateString("en-US", {
                          weekday: "short",
                          month: "short",
                          day: "numeric",
                          year: "numeric",
                        });
                      })()}{" "}
                      at{" "}
                      {(() => {
                        const time = selectedAppointment.original.startTime;
                        const [hours, minutes] = time.split(":");
                        const hour = parseInt(hours);
                        const ampm = hour >= 12 ? "PM" : "AM";
                        const displayHour = hour % 12 || 12;
                        return `${displayHour}:${minutes} ${ampm}`;
                      })()}
                    </p>
                    <p className="body-text-sm text-gray-600">
                      {selectedAppointment.display.venueName}
                    </p>
                    <p className="body-text-xs text-gray-500">
                      Booking ID: {selectedAppointment.display.id}
                    </p>
                  </div>
                </div>

                {/* Reschedule Booking */}
                <button
                  onClick={() => {
                    // TODO: Implement reschedule functionality
                    setShowManageBookingModal(false);
                    // Navigate to reschedule page or open reschedule modal
                  }}
                  className="w-full flex items-center gap-3 p-4 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors text-left"
                >
                  <div className="w-10 h-10 rounded-full bg-purple-100 flex items-center justify-center flex-shrink-0">
                    <RotateCcw className="w-5 h-5 text-purple-600" />
                  </div>
                  <div className="flex-1 flex items-center justify-between">
                    <div>
                      <h4 className="body-text-sm font-medium text-gray-900 mb-1">
                        Reschedule booking
                      </h4>
                      <p className="body-text-xs text-gray-600">
                        Change the date or time of your appointment
                      </p>
                    </div>
                    <ChevronRight className="w-5 h-5 text-gray-400 flex-shrink-0" />
                  </div>
                </button>

                {/* Cancel Booking */}
                <button
                  onClick={() => {
                    setShowManageBookingModal(false);
                    setShowCancelConfirmationModal(true);
                  }}
                  className="w-full flex items-center gap-3 p-4 border border-red-200 rounded-lg hover:bg-red-50 transition-colors text-left"
                >
                  <div className="w-10 h-10 rounded-full bg-red-100 flex items-center justify-center flex-shrink-0">
                    <X className="w-5 h-5 text-red-600" />
                  </div>
                  <div className="flex-1 flex items-center justify-between">
                    <div>
                      <h4 className="body-text-sm font-medium text-red-900 mb-1">
                        Cancel booking
                      </h4>
                      <p className="body-text-xs text-red-600">
                        Cancel this appointment
                        {appointmentDetail.appointment.cancellationCharges &&
                        parseFloat(
                          appointmentDetail.appointment.cancellationCharges
                        ) > 0
                          ? ` (${appointmentDetail.currencyUnit}${parseFloat(
                              appointmentDetail.appointment.cancellationCharges
                            ).toFixed(2)} cancellation fee)`
                          : ""}
                      </p>
                    </div>
                    <ChevronRight className="w-5 h-5 text-red-400 flex-shrink-0" />
                  </div>
                </button>
              </div>
            )}
          </div>
        </div>
      </Dialog>

      {/* Cancel Confirmation Modal */}
      <Dialog
        open={showCancelConfirmationModal}
        onOpenChange={setShowCancelConfirmationModal}
        showCloseButton={true}
        bottomSheet={true}
      >
        <div className="flex flex-col h-full max-w-2xl w-full min-h-0">
          {/* Modal Header */}
          <div className="flex items-center justify-between p-6 pb-4 flex-shrink-0 border-b border-gray-200">
            <h2 className="text-2xl font-bold text-gray-900">
              Cancel booking
            </h2>
            <button
              onClick={() => setShowCancelConfirmationModal(false)}
              className="p-1 rounded-full hover:bg-gray-100 transition-colors"
            >
              <X className="w-5 h-5 text-gray-600" />
            </button>
          </div>

          {/* Modal Content */}
          <div className="flex-1 overflow-y-auto px-6 py-4 min-h-0">
            {selectedAppointment && appointmentDetail && (
              <div className="space-y-4">
                <p className="body-text text-gray-700">
                  Are you sure you want to cancel this booking?
                </p>

                {/* Booking Details */}
                <div className="p-4 bg-gray-50 rounded-lg border border-gray-200">
                  <div className="space-y-2">
                    <p className="body-text-sm font-medium text-gray-900">
                      {selectedAppointment.display.venueName}
                    </p>
                    <p className="body-text-sm text-gray-600">
                      {(() => {
                        const date = new Date(
                          `${selectedAppointment.original.on}T${selectedAppointment.original.startTime}`
                        );
                        return date.toLocaleDateString("en-US", {
                          weekday: "short",
                          month: "short",
                          day: "numeric",
                          year: "numeric",
                        });
                      })()}{" "}
                      at{" "}
                      {(() => {
                        const time = selectedAppointment.original.startTime;
                        const [hours, minutes] = time.split(":");
                        const hour = parseInt(hours);
                        const ampm = hour >= 12 ? "PM" : "AM";
                        const displayHour = hour % 12 || 12;
                        return `${displayHour}:${minutes} ${ampm}`;
                      })()}
                    </p>
                  </div>
                </div>

                {/* Cancellation Charges */}
                {appointmentDetail.appointment.cancellationCharges &&
                  parseFloat(
                    appointmentDetail.appointment.cancellationCharges
                  ) > 0 && (
                    <div className="p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                      <p className="body-text-sm text-yellow-800">
                        <strong>Cancellation fee:</strong>{" "}
                        {appointmentDetail.currencyUnit}
                        {parseFloat(
                          appointmentDetail.appointment.cancellationCharges
                        ).toFixed(2)}
                      </p>
                    </div>
                  )}

                {/* Error Message */}
                {cancelError && (
                  <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
                    <p className="body-text-sm text-red-600">{cancelError}</p>
                  </div>
                )}
              </div>
            )}
          </div>

          {/* Action Buttons */}
          <div className="flex gap-3 p-6 pt-4 border-t border-gray-200 bg-white flex-shrink-0">
            <Button
              variant="secondary"
              className="flex-1"
              onClick={() => setShowCancelConfirmationModal(false)}
              disabled={cancelling}
            >
              No, keep booking
            </Button>
            <Button
              variant="primary"
              className="flex-1 bg-red-600 hover:bg-red-700"
              onClick={handleCancelAppointment}
              disabled={cancelling}
            >
              {cancelling ? "Cancelling..." : "Yes, cancel booking"}
            </Button>
          </div>
        </div>
      </Dialog>
    </div>
  );
}
