"use client";

import { useState, useEffect, Suspense } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import { Eye, EyeOff, ArrowLeft } from "lucide-react";
import { AuthLayoutSplit } from "@/features/auth/components/AuthLayoutSplit";
import { Input } from "@/components/ui/input";
import { Select } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { Button } from "@/components/ui/button";
import { AuthFooterLinks } from "@/features/auth/components/AuthFooterLinks";
import { cn } from "@/lib/classNames";
import {
  getSocialAuthData,
  clearSocialAuthData,
} from "@/lib/socialAuthStorage";
import { routes } from "@/config/routes";
import Link from "next/link";

// Country codes for phone number
const countryCodes = [
  { value: "+1", label: "+1 (US/CA)" },
  { value: "+44", label: "+44 (UK)" },
  { value: "+92", label: "+92 (PK)" },
  { value: "+971", label: "+971 (AE)" },
  { value: "+33", label: "+33 (FR)" },
  { value: "+49", label: "+49 (DE)" },
  { value: "+39", label: "+39 (IT)" },
  { value: "+34", label: "+34 (ES)" },
  { value: "+61", label: "+61 (AU)" },
  { value: "+31", label: "+31 (NL)" },
];

function CreateAccountPageContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const emailParam = searchParams.get("email") || "";
  const isSocialAuthParam = searchParams.get("socialAuth") === "true";

  // Get social auth data from sessionStorage if available
  const [socialAuthData, setSocialAuthDataState] = useState<any>(null);

  // Initialize form data - will be updated when social auth data loads
  const [formData, setFormData] = useState({
    firstName: "",
    lastName: "",
    password: "",
    countryCode: "+1",
    phoneNumber: "",
  });

  const [showPassword, setShowPassword] = useState(false);
  const [agreedToTerms, setAgreedToTerms] = useState(false);
  const [agreedToMarketing, setAgreedToMarketing] = useState(false);
  const [touched, setTouched] = useState<Record<string, boolean>>({});
  const [errors, setErrors] = useState<Record<string, string>>({});
  const [isLoading, setIsLoading] = useState(false);
  const [submitError, setSubmitError] = useState("");
  const [isLoadingSocialData, setIsLoadingSocialData] =
    useState(isSocialAuthParam);

  useEffect(() => {
    if (isSocialAuthParam) {
      const data = getSocialAuthData();
      if (data) {
        setSocialAuthDataState(data);
        // Pre-fill form with social auth data
        setFormData((prev) => ({
          ...prev,
          firstName: data.firstName || "",
          lastName: data.lastName || "",
          phoneNumber: data.phoneNumber || "",
        }));
        setIsLoadingSocialData(false);
      } else {
        // If no social auth data found but param is set, redirect back to signup
        router.push(routes.signup);
      }
    } else if (emailParam) {
      // For email signup, email is already in URL
      // Form fields start empty for user to fill
      setIsLoadingSocialData(false);
    } else {
      setIsLoadingSocialData(false);
    }
  }, [isSocialAuthParam, emailParam, router]);

  // Use social auth data if available, otherwise use URL params (for email signup)
  // These values update when socialAuthData is loaded
  const email = socialAuthData?.email || emailParam;
  const prefillImage = socialAuthData?.image || "";
  const signedBy =
    socialAuthData?.signedBy ||
    (searchParams.get("signedBy") as "google" | "apple" | "email" | null);
  const isSocialAuth =
    isSocialAuthParam || signedBy === "google" || signedBy === "apple";

  // Import validation functions
  const validateField = (name: string, value: string | boolean) => {
    switch (name) {
      case "firstName":
        if (
          !value ||
          (typeof value === "string" && value.trim().length === 0)
        ) {
          return "First name is required";
        }
        if (typeof value === "string" && value.trim().length < 2) {
          return "First name must be at least 2 characters";
        }
        break;
      case "lastName":
        if (
          !value ||
          (typeof value === "string" && value.trim().length === 0)
        ) {
          return "Last name is required";
        }
        if (typeof value === "string" && value.trim().length < 2) {
          return "Last name must be at least 2 characters";
        }
        break;
      case "password":
        // Password is optional for social auth
        if (isSocialAuth) {
          // For social auth, password is not required
          return "";
        }
        if (!value || (typeof value === "string" && value.length === 0)) {
          return "Password is required";
        }
        if (typeof value === "string" && value.length < 8) {
          return "Password must be at least 8 characters";
        }
        if (typeof value === "string") {
          const hasLetter = /[a-zA-Z]/.test(value);
          const hasNumber = /[0-9]/.test(value);
          if (!hasLetter || !hasNumber) {
            return "Password must contain at least one letter and one number";
          }
        }
        break;
      case "phoneNumber":
        if (
          !value ||
          (typeof value === "string" && value.trim().length === 0)
        ) {
          return "Mobile number is required";
        }
        if (typeof value === "string") {
          const cleaned = value.replace(/[\s\-\(\)]/g, "");
          if (!/^\d+$/.test(cleaned)) {
            return "Phone number must contain only digits";
          }
          if (cleaned.length < 7) {
            return "Phone number is too short";
          }
        }
        break;
      case "agreedToTerms":
        if (!value) {
          return "You must agree to the terms and conditions";
        }
        break;
    }
    return "";
  };

  const handleBlur = (fieldName: string) => {
    setTouched((prev) => ({ ...prev, [fieldName]: true }));

    let value: string | boolean;
    if (fieldName === "agreedToTerms") {
      value = agreedToTerms;
    } else {
      value = formData[fieldName as keyof typeof formData] as string;
    }

    const error = validateField(fieldName, value);
    setErrors((prev) => ({
      ...prev,
      [fieldName]: error,
    }));
  };

  const handleChange = (fieldName: string, value: string | boolean) => {
    if (fieldName === "agreedToTerms") {
      setAgreedToTerms(value as boolean);
    } else {
      setFormData((prev) => ({ ...prev, [fieldName]: value }));
    }

    // Clear error when user starts typing
    if (touched[fieldName]) {
      const error = validateField(fieldName, value);
      setErrors((prev) => ({
        ...prev,
        [fieldName]: error,
      }));
    }
  };

  const isFormValid =
    formData.firstName &&
    formData.lastName &&
    formData.phoneNumber &&
    agreedToTerms &&
    (isSocialAuth || formData.password) && // Password only required for email signup
    Object.values(errors).every((error) => !error);

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();

    // Mark all fields as touched
    const allFields = isSocialAuth
      ? ["firstName", "lastName", "phoneNumber", "agreedToTerms"]
      : ["firstName", "lastName", "password", "phoneNumber", "agreedToTerms"];
    const newTouched: Record<string, boolean> = {};
    const newErrors: Record<string, string> = {};

    allFields.forEach((field) => {
      newTouched[field] = true;
      let value: string | boolean;
      if (field === "agreedToTerms") {
        value = agreedToTerms;
      } else {
        value = formData[field as keyof typeof formData] as string;
      }
      const error = validateField(field, value);
      if (error) {
        newErrors[field] = error;
      }
    });

    setTouched(newTouched);
    setErrors(newErrors);

    if (Object.keys(newErrors).length === 0) {
      // Handle account creation via API
      handleSignup();
    }
  };

  const handleSignup = async () => {
    setIsLoading(true);
    try {
      const { signup } = await import("@/features/auth/api/signup");
      const { checkEmail } = await import("@/features/auth/api/emailChecker");
      const { setAccessToken, setUserData } = await import("@/lib/auth");

      const response = await signup({
        firstName: formData.firstName.trim(),
        lastName: formData.lastName.trim(),
        email: email.trim(),
        countryCode: formData.countryCode,
        phoneNum: formData.phoneNumber.replace(/[\s\-\(\)]/g, ""),
        password: isSocialAuth
          ? "social-auth-temp-password"
          : formData.password,
        signedBy: signedBy || "email",
        image: prefillImage,
      });

      // Clear social auth data after successful signup
      if (isSocialAuth) {
        clearSocialAuthData();
      }

      // For social auth, skip OTP verification
      if (isSocialAuth) {
        // Check if API returned accessToken directly (some APIs do this for social auth)
        if (response.accessToken) {
          setAccessToken(response.accessToken);
          if (response.userId || response.email) {
            setUserData({
              accessToken: response.accessToken,
              userId: response.userId?.toString() || "",
              email: email.trim(),
              firstName: formData.firstName.trim(),
              lastName: formData.lastName.trim(),
              phoneNum: formData.phoneNumber.replace(/[\s\-\(\)]/g, ""),
              image: prefillImage || "",
              stripeCustomerId: null,
            });
          }
          router.push(routes.home);
          return;
        }

        // If no accessToken in response, check email to get access token
        // Wait a moment for the account to be created
        await new Promise((resolve) => setTimeout(resolve, 1500));

        try {
          const emailCheck = await checkEmail(email.trim());

          if (emailCheck.status === "1" || emailCheck.status === "2") {
            const userData = emailCheck.data as any;
            if (userData && userData.accessToken) {
              setAccessToken(userData.accessToken);
              setUserData(userData);
              router.push(routes.home);
              return;
            }
          }
        } catch (checkError) {
          console.error("Email check error:", checkError);
        }

        // If we can't get access token, still skip OTP and go to home
        // User can log in again if needed
        router.push(routes.home);
      } else {
        // For email signup, navigate to OTP verification page
        if (response.otpId && response.userId) {
          router.push(
            `/signup/verify-otp?otpId=${response.otpId}&userId=${
              response.userId
            }&email=${encodeURIComponent(email)}`
          );
        } else {
          setSubmitError("Invalid response from server. Please try again.");
        }
      }
    } catch (error: any) {
      console.error("Signup error:", error);
      setSubmitError(
        error.message || "Failed to create account. Please try again."
      );
    } finally {
      setIsLoading(false);
    }
  };

  // Show loading state while social auth data is being loaded
  if (isLoadingSocialData) {
    return (
      <AuthLayoutSplit>
        <div className="max-w-md mx-auto w-full flex flex-col min-h-full items-center justify-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-gray-900"></div>
        </div>
      </AuthLayoutSplit>
    );
  }

  return (
    <AuthLayoutSplit>
      <div className="max-w-md mx-auto w-full flex flex-col min-h-full">
        <div className="mb-8">
          <button
            onClick={() => router.back()}
            className="mb-6 p-2 -ml-2 hover:bg-gray-50 rounded-lg transition-colors"
            aria-label="Go back"
          >
            <ArrowLeft className="w-5 h-5 text-gray-900" />
          </button>
          <h1 className="heading-2 mb-2">Create account</h1>
          {email && (
            <p className="body-text-muted">
              {isSocialAuth
                ? `Complete your account setup for ${email}. Please provide your phone number to continue.`
                : `You're almost there! Create your new account for ${email} by completing these details.`}
            </p>
          )}
        </div>

        <form onSubmit={handleSubmit} className="flex flex-col flex-1">
          <div className="space-y-4 mb-6">
            {/* First Name */}
            <Input
              id="firstName"
              label="First name *"
              value={formData.firstName}
              onChange={(e) => handleChange("firstName", e.target.value)}
              onBlur={() => handleBlur("firstName")}
              error={errors.firstName}
              touched={touched.firstName}
              required
            />

            {/* Last Name */}
            <Input
              id="lastName"
              label="Last name *"
              value={formData.lastName}
              onChange={(e) => handleChange("lastName", e.target.value)}
              onBlur={() => handleBlur("lastName")}
              error={errors.lastName}
              touched={touched.lastName}
              required
            />

            {/* Password - Only show for email signup */}
            {!isSocialAuth && (
              <div>
                <label
                  className={cn(
                    "label block mb-2",
                    touched.password && errors.password && "text-error-500"
                  )}
                >
                  Password *
                </label>
                <div className="relative">
                  <Input
                    id="password"
                    type={showPassword ? "text" : "password"}
                    value={formData.password}
                    onChange={(e) => handleChange("password", e.target.value)}
                    onBlur={() => handleBlur("password")}
                    placeholder="Enter your password"
                    error={errors.password}
                    touched={touched.password}
                    required
                    className="pr-12"
                  />
                  <button
                    type="button"
                    onClick={() => setShowPassword(!showPassword)}
                    className="absolute right-4 top-1/2 -translate-y-1/2 text-gray-500 hover:text-gray-700 z-10"
                  >
                    {showPassword ? (
                      <EyeOff className="w-5 h-5" />
                    ) : (
                      <Eye className="w-5 h-5" />
                    )}
                  </button>
                </div>
              </div>
            )}

            {/* Mobile Number */}
            <div>
              <label
                className={cn(
                  "label block mb-2",
                  touched.phoneNumber && errors.phoneNumber && "text-error-500"
                )}
              >
                Mobile number *
              </label>
              <div className="flex gap-3">
                <Select
                  value={formData.countryCode}
                  onChange={(e) =>
                    setFormData({ ...formData, countryCode: e.target.value })
                  }
                  className="w-32"
                >
                  {countryCodes.map((code) => (
                    <option key={code.value} value={code.value}>
                      {code.label}
                    </option>
                  ))}
                </Select>
                <Input
                  id="phoneNumber"
                  type="tel"
                  value={formData.phoneNumber}
                  onChange={(e) => handleChange("phoneNumber", e.target.value)}
                  onBlur={() => handleBlur("phoneNumber")}
                  placeholder="Phone number"
                  error={errors.phoneNumber}
                  touched={touched.phoneNumber}
                  required
                  className="flex-1"
                />
              </div>
            </div>
          </div>

          {/* Checkboxes */}
          <div className="space-y-4 mb-6">
            <Checkbox
              id="agreedToTerms"
              checked={agreedToTerms}
              onChange={(e) => handleChange("agreedToTerms", e.target.checked)}
              onBlur={() => handleBlur("agreedToTerms")}
              error={errors.agreedToTerms}
              touched={touched.agreedToTerms}
              required
              label={
                <>
                  I agree to the{" "}
                  <Link
                    href="/privacy"
                    className="text-primary-500 hover:underline"
                  >
                    Privacy Policy
                  </Link>
                  ,{" "}
                  <Link
                    href="/terms"
                    className="text-primary-500 hover:underline"
                  >
                    Terms of Use
                  </Link>{" "}
                  and{" "}
                  <Link
                    href="/terms-of-service"
                    className="text-primary-500 hover:underline"
                  >
                    Terms of Service
                  </Link>
                  .
                </>
              }
            />

            <Checkbox
              checked={agreedToMarketing}
              onChange={(e) => setAgreedToMarketing(e.target.checked)}
              label="I agree to receive marketing notifications with offers and news."
            />
          </div>

          {/* Submit Error */}
          {submitError && (
            <div className="mb-4 p-3 rounded-xl bg-error-50 border border-error-200">
              <p className="body-text-sm text-error-500">{submitError}</p>
            </div>
          )}

          {/* Submit Button */}
          <Button
            type="submit"
            disabled={!isFormValid}
            loading={isLoading}
            className="w-full mb-auto"
          >
            Continue
          </Button>

          {/* Footer Links */}
          <AuthFooterLinks />
        </form>
      </div>
    </AuthLayoutSplit>
  );
}

export default function CreateAccountPage() {
  return (
    <Suspense fallback={
      <AuthLayoutSplit>
        <div className="max-w-md mx-auto w-full flex flex-col min-h-full items-center justify-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-gray-900"></div>
        </div>
      </AuthLayoutSplit>
    }>
      <CreateAccountPageContent />
    </Suspense>
  );
}
